/* tc-coffee.c -- Assembler for the COFFEE RISC
   Copyright 2005, 2006 Free Software Foundation, Inc.

   This file is part of GAS, the GNU Assembler.

   GAS is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   GAS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GAS; see the file COPYING.  If not, write to the Free
   Software Foundation, 51 Franklin Street - Fifth Floor, Boston, MA
   02110-1301, USA.  */

#include "as.h"
#include "safe-ctype.h"
//#include "subsegs.h"
#include "dwarf2dbg.h"
#include "opcode/coffee.h"
//#include "elf/coffee.h"

#define COFFEE_DEBUG 0

/* These chars start a comment anywhere in a source file (except inside
   another comment).  */
const char comment_chars[] = ";";

/* These chars only start a comment at the beginning of a line.  */
const char line_comment_chars[] = "#";

/* This is a null terminated array of characters which separate lines (null
   and newline are such characters by default, and need not be listed in this
   array).  Note that line_separator_chars do not separate lines if found in
   a comment, such as after a character in line_comment_chars or
   comment_chars.  */
const char line_separator_chars[] = "$";

/* This is a null terminated array of characters which may be used as the
   exponent character in a floating point number.  This is normally "eE".  */
const char EXP_CHARS[] = "eE";

/* This is a null terminated array of characters which may be used to indicate
   a floating point constant.  A zero followed by one of these characters is
   assumed to be followed by a floating point number; thus they operate the
   way that '0x' is used to indicate a hexadecimal constant.  */
const char FLT_CHARS[] = "fFdD";

/* This is a const array of type `pseudo_typeS'.  It is a mapping from
   pseudo-op names to functions.  You should use this table to implement
   pseudo-ops which are specific to the CPU.  */
const pseudo_typeS md_pseudo_table[] =
{
  {"code16", s_ignore, 0},
  {"code32", s_ignore, 0},
  {"proc", s_ignore, 0},
  {"endproc", s_ignore, 0},
  {"word", cons, 4},
  {NULL, NULL, 0}
};

/* Flag to enable support for Milk coprocessor when non-zero.  */
static unsigned int milk_enabled = 0;

/* Handle to the opcode hash table.  */
static struct hash_control *op_hash = NULL;

/* Description of the instruction being assembled.  */
struct coffee_insn
{
  unsigned long opcode;
  int size;
  struct
  {
    bfd_reloc_code_real_type type;
    expressionS expr;
    int pcrel;
  } reloc;
};

/* Prototypes for internal functions.  */
static void coffee_output_insn (struct coffee_insn *);
static int coffee_parse_operands (char *, struct coffee_opcode *, struct coffee_insn *);

/* Return non-zero if VAL is in the range -(MAX+1) to MAX.  */
static inline int
in_signed_range (long val, long max)
{
  assert (max > 0);
  val = ((val & 0xffffffff) ^ (1 << 31)) - (1 << 31);
  if (val > max)
    return 0;
  if (val < ~max)
    return 0;
  return 1;
}

/* Return non-zero if VAL is in the range 0 to MAX.  */
static inline int
in_unsigned_range (long val, long max)
{
  assert (max > 0);
  if (val > max)
    return 0;
  return 1;
}

/* Return non-zero if VAL is in the range -(MAX/2+1) to MAX.
   (e.g. -15 to +31).  */
static inline int
in_bitfield_range (long val, long max)
{
  assert (max > 0);
  if (val > max)
    return 0;
  if (val < ~(max >> 1))
    return 0;
  return 1;
}

/* --- Start of meta-instruction implementations. --- */

static void
coffee_meta_incdec (char *str, char mode)
{
#if 0
  printf ("COFFEE_DEBUG: meta_incdec(\"%s\",'%c')\n", str, mode);
#endif
  long val;
  expressionS exp;
  struct coffee_insn insn;
  struct coffee_opcode *format;
  format = (struct coffee_opcode *) hash_find (op_hash, "addi");

  memset (&insn, 0, sizeof (insn));
  insn.opcode = format->opcode;
  insn.size = 4;
  insn.reloc.type = BFD_RELOC_NONE;

  input_line_pointer = str;
  expression (&exp);
  if (exp.X_op != O_register)
    {
      as_bad (_("register operand expected: %s"), str);
      return;
    }
  else
    {
      val = (long) exp.X_add_number;
      if (val < 0 || val > MAX_REG)
        {
          as_bad (_("invalid destination register: %s"), str);
          return;
        }
      else
        {
          insn.opcode |= val;
          insn.opcode |= val << 5;
          insn.opcode |= (((TOLOWER(mode) == 'd') ? -1 : 1) & 0x7fff) << 10;
        }
    }
  coffee_output_insn (&insn);

  if (ISLOWER(mode))
    {
      format = (struct coffee_opcode *) hash_find (op_hash, "andi");
      insn.opcode = format->opcode;
      
      insn.opcode |= val;
      insn.opcode |= val << 5;
      insn.opcode |= 0xff << 10;
      
      coffee_output_insn (&insn);
    }
}

static inline void
coffee_meta_dec (char *str)
{
  coffee_meta_incdec (str, 'D');
}

static inline void
coffee_meta_decb (char *str)
{
  coffee_meta_incdec (str, 'd');
}

static inline void
coffee_meta_inc (char *str)
{
  coffee_meta_incdec (str, 'I');
}

static inline void
coffee_meta_incb (char *str)
{
  coffee_meta_incdec (str, 'i');
}

static void
coffee_meta_ldra (char *str)
{
#if 0
  printf ("COFFEE_DEBUG: meta_ldra(\"%s\")\n", str);
#endif
  long dreg;
  expressionS exp;
  struct coffee_opcode *format;
  struct coffee_insn insn;
  memset (&insn, 0, sizeof (insn));

  input_line_pointer = str;
  expression (&exp);
  if (exp.X_op != O_register)
    {
      as_bad (_("register operand expected: %s"), str);
      return;
    }
  else
    {
      dreg = /* (long) */ exp.X_add_number;
      if (dreg < 0 || dreg > MAX_REG)
        {
          as_bad (_("invalid destination register: %s"), str);
          return;
        }
    }

  if (*input_line_pointer++ != ',')
    {
      as_bad (_("bad operands: %s"), str);
      return;
    }
  
  expression (&exp);
  if (exp.X_op != O_symbol)
    {
      as_bad (_("address operand expected: %s"), str);
      return;
    }

  format = (struct coffee_opcode *) hash_find (op_hash, "lli");
  insn.opcode = format->opcode;
  insn.size = 4;
  insn.reloc.type = BFD_RELOC_LO16;
  insn.reloc.expr = exp;
  insn.reloc.pcrel = 0;
  insn.opcode |= dreg /* << 0 */;
  coffee_output_insn (&insn);

  format = (struct coffee_opcode *) hash_find (op_hash, "lui");
  insn.opcode = format->opcode;
  insn.size = 4;
  insn.reloc.type = BFD_RELOC_HI16;
  insn.reloc.expr = exp;
  insn.reloc.pcrel = 0;
  insn.opcode |= dreg /* << 0 */;
  coffee_output_insn (&insn);
}

static void
coffee_meta_ldri (char *str)
{
#if 0
  printf ("COFFEE_DEBUG: meta_ldri(\"%s\")\n", str);
#endif
  long dreg, val;
  expressionS exp;
  struct coffee_opcode *format;
  struct coffee_insn insn;
  memset (&insn, 0, sizeof (insn));
  insn.size = 4;
  insn.reloc.type = BFD_RELOC_NONE;

  input_line_pointer = str;
  expression (&exp);
  if (exp.X_op != O_register)
    {
      as_bad (_("register operand expected: %s"), str);
      return;
    }
  else
    {
      dreg = /* (long) */ exp.X_add_number;
      if (dreg < 0 || dreg > MAX_REG)
        {
          as_bad (_("invalid destination register: %s"), str);
          return;
        }
    }

  if (*input_line_pointer++ != ',')
    {
      as_bad (_("bad operands: %s"), str);
      return;
    }
  
  val = get_absolute_expression ();

  format = (struct coffee_opcode *) hash_find (op_hash, "lli");
  insn.opcode = format->opcode;
  insn.opcode |= dreg /* << 0 */;
  insn.opcode |= ((val >> 15) & 0x1) << 9;
  insn.opcode |= (val & 0x7fff) << 10;
  coffee_output_insn (&insn);

  if (val >>= 16)
    {
      format = (struct coffee_opcode *) hash_find (op_hash, "lui");
      insn.opcode = format->opcode;
      insn.opcode |= dreg /* << 0 */;
      insn.opcode |= ((val >> 15) & 0x1) << 9;
      insn.opcode |= (val & 0x7fff) << 10;
      coffee_output_insn (&insn);
    }
}

struct coffee_meta
{
  const char *name;
  void (*function) (char *);
};
const struct coffee_meta coffee_metas[] =
{
  {"dec",  coffee_meta_dec},
  {"decb", coffee_meta_decb},
  {"inc",  coffee_meta_inc},
  {"incb", coffee_meta_incb},
  {"ldra", coffee_meta_ldra},
  {"ldri", coffee_meta_ldri},
  {NULL, NULL}
};

/* --- End of meta-instruction implementations. --- */


/* GAS will call this function at the start of the assembly, after the command
   line arguments have been parsed and all the machine independent
   initializations have been completed.  */

void
md_begin (void)
{
  const char *retval = NULL;
  unsigned int i = 0;

  /* Set up a hash table for the instructions.  */
  if (!(op_hash = hash_new ()))
    as_fatal (_("Internal error: failed hash creation"));

  while (coffee_opcodes[i].name)
    {
      const char *name = coffee_opcodes[i].name;
      retval = hash_insert (op_hash,
                            coffee_opcodes[i].name,
                            (PTR) &coffee_opcodes[i]);
      if (retval != NULL)
        as_fatal (_("Can't hash instruction `%s': %s"),
                  coffee_opcodes[i].name, retval);

      /* Insert only unique names into hash table.  */
      do
        {
          //if (coffee_opcodes[i].match & coffee_opcodes[i].lose)
          //  as_fatal (_("Internal error: losing opcode: `%s' \"%s\""),
          //            coffee_opcodes[i].name, coffee_opcodes[i].args);
          ++i;
        }
      while (coffee_opcodes[i].name
             && strcmp (coffee_opcodes[i].name, name) == 0);
    }

  /* Initialize register symbols to simplify argument parsing.  */
  for (i = 0; coffee_regs[i].name; ++i)
    {
      if (symbol_find (coffee_regs[i].name))
        continue;
      /* Use symbol_create here instead of symbol_new so we don't try to
         output registers into the object file's symbol table.  */
      symbol_table_insert (symbol_create (coffee_regs[i].name,
                                          reg_section,
                                          (valueT) coffee_regs[i].value,
                                          &zero_address_frag));
    }

  /* Optionally add the Milk opcodes and registers.  */
  if (milk_enabled)
    {
      i = 0;
      while (milk_opcodes[i].name)
        {
          const char *name = milk_opcodes[i].name;
          retval = hash_insert (op_hash,
                                milk_opcodes[i].name,
                                (PTR) &milk_opcodes[i]);
          if (retval != NULL)
            as_fatal (_("Can't hash instruction `%s': %s"),
                      milk_opcodes[i].name, retval);
          do
            {
              ++i;
            }
          while (milk_opcodes[i].name
                 && strcmp (milk_opcodes[i].name, name) == 0);
        }

      for (i = 0; milk_regs[i].name; ++i)
        {
          if (symbol_find (milk_regs[i].name))
            continue;
          symbol_table_insert (symbol_create (milk_regs[i].name,
                                              reg_section,
                                              (valueT) milk_regs[i].value,
                                              &zero_address_frag));
        }
    }

#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG: md_begin()\n");
  if (milk_enabled)
    printf ("COFFEE_DEBUG:\tmilk_enabled\n");
#endif
}

/* GAS will call this function for each input line which does not contain a
   pseudo-op.  The argument is a null terminated string.  The function should
   assemble the string as an instruction with operands.  Normally `md_assemble'
   will do this by calling `frag_more' and writing out some bytes.
   `md_assemble' will call `fix_new' to create fixups as needed.
   Targets which need to do special purpose relaxation will call `frag_var'.  */

void
md_assemble (char *str)
{
  struct coffee_opcode *format;
  struct coffee_insn insn;
  unsigned int i;
  char *param;

  know (str);

#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG: md_assemble(\"%s\")\n", str);
#endif

  /* Get the opcode.  */
  for (param = str; *param != '\0' && ! ISSPACE (*param); param++)
    ;
  if (*param != '\0')
    *param++ = '\0';

  /* Look up the opcode in the hash table.  */
  format = (struct coffee_opcode *) hash_find (op_hash, str);
  if (!format)
    {
      /* It wasn't a regular instruction so check for a meta-instruction next.  */
      for (i = 0; coffee_metas[i].name; ++i)
        {
          if (strcmp (str, coffee_metas[i].name) == 0)
            {
              /* Make it so, if we have a match.  */
              coffee_metas[i].function (param);
              return;
            }
        }
        /* No luck.  Give up.  */
        as_bad (_("Unknown opcode '%s'"), str);
        return;
    }

  /* Initialize output_insn.  */
  memset (&insn, 0, sizeof (insn));
  insn.opcode = format->opcode;
  insn.size = 4;
  insn.reloc.type = BFD_RELOC_NONE;

  /* Do the low-level grunt - assemble operands to bits.  */
  if (! coffee_parse_operands (param, format, &insn))
    {
#if COFFEE_DEBUG
      as_tsktsk ("... error terror ...");
#endif
      return;
    }
  
  /* Output the instruction opcode.  */
  coffee_output_insn (&insn);
}

/* Subroutine of md_assemble to output the insn.  */

static void
coffee_output_insn (struct coffee_insn *insn)
{
  /* Create space for the opcode.  */
  char *toP = frag_more (insn->size);

  /* Put out the opcode.  */
  md_number_to_chars (toP, (valueT) insn->opcode, insn->size);

#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG: output_insn(0x%lX/%i)\n", insn->opcode, insn->size);
#endif

  /* Put out the symbol-dependent stuff.  */
  if (insn->reloc.type != BFD_RELOC_NONE)
    {
#if COFFEE_DEBUG
      printf ("COFFEE_DEBUG:\treloc.type=%s, reloc.expr=%u, reloc.pcrel=%i\n",
              bfd_get_reloc_code_name(insn->reloc.type),
              insn->reloc.expr.X_op, insn->reloc.pcrel);
#endif

      fix_new_exp (frag_now,
                   (toP - frag_now->fr_literal),
                   insn->size,
                   &insn->reloc.expr,
                   insn->reloc.pcrel,
                   insn->reloc.type);

      //fixP->fx_no_overflow = 1;
    }

  /* Put out the debug-related information.  */
  dwarf2_emit_insn (insn->size);
}


/* Subroutine of md_assemble for low level text-to-bits assembly of operands.
   The return value 0 indicates an error.  */

static int
coffee_parse_operands (char *param,
                       struct coffee_opcode *format,
                       struct coffee_insn *insn)
{
  const char *arg;
  
  for (arg = format->args; *arg != '\0'; ++arg)
    {
      offsetT val = 0;
      char *save;
      expressionS exp;

      if (*arg == ',')
        {
          /* This must match exactly, and not be the last in line.  */
          if ((*param++ != *arg++) || *arg == '\0')
            {
              as_bad (_("bad operands: %s"), --param);
              return 0;
            }
        }

      switch (*arg)
        {
        case 'r':
          /* Parse the operand, trying to find a register.  */
          save = input_line_pointer;
          input_line_pointer = param;
          expression (&exp);
          param = input_line_pointer;
          input_line_pointer = save;

#if 0
          printf ("COFFEE_DEBUG:\targ=%c%c, exp.X_op=%u, exp.X_add_number=%li\n",
                  *arg, *(arg+1), exp.X_op, exp.X_add_number);
#endif

          if (exp.X_op == O_register)
            {
              val = exp.X_add_number;
            }
          else
            {
              as_bad (_("register operand expected"));
              return 0;
            }

          /* The type of a register operand follows the 'r'.  */
          switch (*++arg)
            {
            case 'D':
              /* Destination register in bits <4:0>.  */
              if (in_unsigned_range(val,MAX_REG))
                {
                  insn->opcode |= val /* << 0 */;
                  continue;
                }
              else
                {
                  as_bad (_("invalid destination register: %s"), save);
                  return 0;
                }

            case 'A':
              /* Source register in bits <9:5>.  */
              if (in_unsigned_range(val,MAX_REG))
                {
                  insn->opcode |= val << 5;
                  continue;
                }
              else
                {
                  as_bad (_("invalid source register: %s"), save);
                  return 0;
                }

            case 'B':
              /* Source register in bits <14:10>.  */
              if (in_unsigned_range(val,MAX_REG))
                {
                  insn->opcode |= val << 10;
                  continue;
                }
              else
                {
                  as_bad (_("invalid source register: %s"), save);
                  return 0;
                }

            case 'C':
              /* Condition register in bits <23:22>.  */
              if (in_unsigned_range(val,MAX_CREG))
                {
                  insn->opcode |= val << 22;
                  continue;
                }
              else
                {
                  as_bad (_("invalid condition register: %s"), save);
                  return 0;
                }
                
            case 'P':
              /* Coprocessor register in bits <16:12>.  */
              if (in_unsigned_range(val,MAX_COPREG))
                {
                  insn->opcode |= val << 12;
                  continue;
                }
              else
                {
                  as_bad (_("invalid coprocessor register: %s"), save);
                  return 0;
                }

            default:
              BAD_CASE (*arg);
            }

        case 'c':
          /* Parse the operand, trying to find a constant (i.e. absolute expression).  */
          save = input_line_pointer;
          input_line_pointer = param;
          val = get_absolute_expression ();
          param = input_line_pointer;
          input_line_pointer = save;

#if 0
          printf ("COFFEE_DEBUG:\targ=%c%c, val=%li\n",
                  *arg, *(arg+1), val);
#endif

          /* The type of a constant operand follows the 'c'.  */
          switch (*++arg)
            {
            case 'I':
              /* 15-bit signed immediate constant in bits <24:10>.  */
              if (in_signed_range(val,0x3fff))
                {
                  insn->opcode |= (val & 0x7fff) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 15-bit signed range: %ld"), val);
                  return 0;
                }

            case 'i':
              /* 9-bit signed immediate constant in bits <18:10>.  */
              if (in_signed_range(val,0xff))
                {
                  insn->opcode |= (val & 0x1ff) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 9-bit signed range: %ld"), val);
                  return 0;
                }

            case 'U':
              /* 15-bit unsigned immediate constant in bits <24:10>.  */
              if (in_unsigned_range(val,0x7fff))
                {
                  insn->opcode |= (val & 0x7fff) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 15-bit unsigned range: %ld"), val);
                  return 0;
                }

            case 'u':
              /* 9-bit unsigned immediate constant in bits <18:10>.  */
              if (in_unsigned_range(val,0x1ff))
                {
                  insn->opcode |= (val & 0x1ff) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 9-bit unsigned range: %ld"), val);
                  return 0;
                }

            case 'C':
              /* 17-bit signed immediate divided in bits <21:10> and <4:0>.  */
              if (in_signed_range(val,0xffff))
                {
                  insn->opcode |= (val & 0xfff) << 10;
                  insn->opcode |= ((val >> 12) & 0x1f) /* << 0 */;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 17-bit range: %ld"), val);
                  return 0;
                }

            case 'S':
              /* 15-bit signed immediate divided in bits <24:15> and <4:0>.  */
              if (in_signed_range(val,0x3fff))
                {
                  insn->opcode |= (val & 0x3ff) << 15;
                  insn->opcode |= ((val >> 10) & 0x1f) /* << 0 */;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 15-bit range: %ld"), val);
                  return 0;
                }

            case 'K':
              /* Small constant of 0..32 in bits <15:10>.  */
              if (in_unsigned_range(val,32))
                {
                  insn->opcode |= (val & 0x3f) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value must be 0-32: %ld"), val);
                  return 0;
                }

            case 'N':
              /* Small constant of 0..3 in bits <11:10>.  */
              if (in_unsigned_range(val,3))
                {
                  insn->opcode |= (val & 0x3) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value must be 0-3: %ld"), val);
                  return 0;
                }

            case 'n':
              /* Small constant of 0 or 1 in bit <10>.  */
              if (in_unsigned_range(val,1))
                {
                  insn->opcode |= (val & 0x1) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value must be 0 or 1: %ld"), val);
                  return 0;
                }

            case 'B':
              /* Bitfield length (0-32) in bits <20:15>.  */
              if (in_unsigned_range(val,32))
                {
                  insn->opcode |= (val & 0x3f) << 15;
                  continue;
                }
              else
                {
                  as_bad (_("bitfield length too long: %ld"), val);
                  return 0;
                }

            case 'b':
              /* Bitfield position (0-31) in bits <14:10>.  */
              if (in_unsigned_range(val,31))
                {
                  insn->opcode |= (val & 0x1f) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("bitfield position out of range: %ld"), val);
                  return 0;
                }

            case 'P':
              /* Coprocessor number in bits <25:24>.  */
              if (in_unsigned_range(val,3))
                {
                  insn->opcode |= (val & 0x3) << 24;
                  continue;
                }
              else
                {
                  as_bad (_("coprocessors over 0-3 not supported: %ld"), val);
                  return 0;
                }

            case 'O':
              /* Coprocessor instruction in bits <23:0>.  */
              if (in_bitfield_range(val,0xffffff))
                {
                  insn->opcode |= (val & 0xffffff) /* << 0 */;
                  continue;
                }
              else
                {
                  as_bad (_("coprocessor instruction overflow: %ld"), val);
                  return 0;
                }
                
            case 'X':
              /* 16-bit bitfield immediate in bits <24:10> except MSB in <9>.  */
              if (in_bitfield_range(val,0xffff))
                {
                  insn->opcode |= ((val >> 15) & 0x1) << 9;
                  insn->opcode |= (val & 0x7fff) << 10;
                  continue;
                }
              else
                {
                  as_bad (_("immediate value not in 16-bit range: %ld"), val);
                  return 0;
                }

            default:
              BAD_CASE (*arg);
            }

        case 'b':
          /* Parse the operand, trying to find a branch target.  */
          save = input_line_pointer;
          input_line_pointer = param;
          expression_and_evaluate (&exp);
          param = input_line_pointer;
          input_line_pointer = save;

#if 0
          printf ("COFFEE_DEBUG:\targ=%c%c, exp.X_op=%u, exp.X_add_number=%li\n",
                  *arg, *(arg+1), exp.X_op, exp.X_add_number);
#endif

          if (exp.X_op == O_constant)
            {
              val = exp.X_add_number;
            }
          else if (exp.X_op == O_illegal || exp.X_op == O_absent || exp.X_op == O_register)
            {
              as_bad (_("branch target operand expected"));
              return 0;
            }
#if COFFEE_DEBUG
          else
            {
              as_tsktsk (_("branch to symbol"));
            }
#endif

          /* The type of a branch target operand follows the 'b'.  */
          switch (*++arg)
            {
            case 'B':
              /* 22-bit branch address in bits <21:0>.  */
              if (exp.X_op == O_constant)
                {
                  if (in_signed_range(val,0x1fffff))
                    {
                      val >>= 1;
                      insn->opcode |= (val & 0x003fffff);
                      continue;
                    }
                  else
                    {
                      as_bad (_("branch target out of range (%ld)"), val);
                      return 0;
                    }
                }
              else
                {
                  insn->reloc.type = BFD_RELOC_COFFEE_BRANCH;
                  insn->reloc.expr = exp;
                  insn->reloc.pcrel = 1;
                  continue;
                }

            case 'J':
              /* 25-bit jump address in bits <24:0>.  */
              if (exp.X_op == O_constant)
                {
                  if (in_signed_range(val,0x7fffff))
                    {
                      val >>= 1;
                      insn->opcode |= (val & 0x01ffffff);
                      continue;
                    }
                  else
                    {
                      as_bad (_("jump target out of range (%ld)"), val);
                      return 0;
                    }
                }
              else
                {
                  insn->reloc.type = BFD_RELOC_COFFEE_JUMP;
                  insn->reloc.expr = exp;
                  insn->reloc.pcrel = 1;
                  continue;
                }

            default:
              BAD_CASE (*arg);
            }

        case 'm':
          /* These types of args are valid only when Milk is enabled.  */
          if (!milk_enabled)
            BAD_CASE (*arg);

          /* Parse the operand, trying to find a register.  */
          save = input_line_pointer;
          input_line_pointer = param;
          expression (&exp);
          param = input_line_pointer;
          input_line_pointer = save;

#if 0
          printf ("COFFEE_DEBUG:\targ=%c%c, exp.X_op=%u, exp.X_add_number=%li\n",
                  *arg, *(arg+1), exp.X_op, exp.X_add_number);
#endif

          if (exp.X_op == O_register)
            {
              val = exp.X_add_number;
            }
          else
            {
              as_bad (_("Milk register operand expected"));
              return 0;
            }

          /* The type of a register operand follows the 'r'.  */
          switch (*++arg)
            {
            case 'D':
              /* Milk result register in bits <10:6>.  */
              if (in_unsigned_range(val,MAX_MILKREG))
                {
                  insn->opcode |= val << 6;
                  continue;
                }
              else
                {
                  as_bad (_("invalid Milk result register: %s"), save);
                  return 0;
                }

            case 'A':
              /* Milk operand register in bits <15:11>.  */
              if (in_unsigned_range(val,MAX_MILKREG))
                {
                  insn->opcode |= val << 11;
                  continue;
                }
              else
                {
                  as_bad (_("invalid Milk operand register: %s"), save);
                  return 0;
                }

            case 'B':
              /* Milk operand register in bits <20:16>.  */
              if (in_unsigned_range(val,MAX_MILKREG))
                {
                  insn->opcode |= val << 16;
                  continue;
                }
              else
                {
                  as_bad (_("invalid Milk operand register: %s"), save);
                  return 0;
                }

            default:
              BAD_CASE (*arg);
            }

        default:
          BAD_CASE (*arg);
        }
    }

  /* We've come to the end of arguments, so we should be out and done cleanly by now.  */
  //demand_empty_rest_of_line ();
  return 1;
}


/* GAS uses these variables and functions during option processing.

   `md_shortopts' is a `const char * which GAS adds to the machine
   independent string passed to `getopt.
   `md_longopts' is a `struct option [] which GAS adds to the machine
   independent long options passed to `getopt; you may use OPTION_MD_BASE,
   defined in "as.h", as the start of a set of long option indices, if necessary.
   `md_longopts_size is a `size_t holding the size of `md_longopts.

   GAS will call `md_parse_option whenever `getopt returns an unrecognized
   code, presumably indicating a special code value which appears in
   `md_longopts.  This function should return non-zero if it handled the
   option and zero otherwise.  There is no need to print a message about an
   option not being recognised.  This will be handled by the generic code.

   GAS will call `md_show_usage when a usage message is printed; it should
   print a description of the machine specific options.

   `md_after_parse_args, if defined, is called after all options are
   processed, to let the backend override settings done by the generic option
   parsing.  */

const char *const md_shortopts = "m";
const struct option md_longopts[] = {
#define OPTION_MILK (OPTION_MD_BASE)
  {"mmilk", no_argument, NULL, OPTION_MILK},
  {NULL, no_argument, NULL, 0}
};
const size_t md_longopts_size = sizeof (md_longopts);

int
md_parse_option (int c, char *arg ATTRIBUTE_UNUSED)
{
  switch (c)
    {
    case 'm':
#if 0
    if (strcmp (arg, "milk") == 0)
        {
          milk_enabled = 1;
        }
      else
        {
          //as_bad (_("invalid switch -m%s"), arg);
          return 0;
        }
      break;
#endif
    case OPTION_MILK:
      milk_enabled = 1;
      break;

    default:
      return 0;
    }

  return 1;
}

void
md_show_usage (FILE *stream)
{
  fprintf (stream, _("\
COFFEE options:\n\
-mmilk			enable support for Milk coprocessor\n"));
}

/* This function is called to convert an ASCII string into a floating point
   value in format used by the CPU.  It takes three arguments.  The first is
   `type which is a byte describing the type of floating point number to be
   created.  Possible values are 'f' or 's' for single precision, 'd' or 'r'
   for double precision and 'x' or 'p' for extended precision.  Either lower
   or upper case versions of these letters can be used.

   The second parameter is `litP which is a pointer to a byte array where the
   converted value should be stored.  The third argument is `sizeP, which is
   a pointer to a integer that should be filled in with the number of
   LITTLENUMs emitted into the byte array.  (LITTLENUM is defined in
   gas/bignum.h).  The function should return NULL upon success or an error
   string upon failure.  */

/* Equal to MAX_PRECISION in atof-ieee.c.  */
#define MAX_LITTLENUMS 6

char *
md_atof (int type, char *litP, int *sizeP)
{
  int prec;
  LITTLENUM_TYPE words[MAX_LITTLENUMS];
  LITTLENUM_TYPE *wordP;
  char *t;

  switch (type)
    {
    case 'f':
    case 'F':
    case 's':
    case 'S':
      prec = 2;
      break;

    case 'd':
    case 'D':
    case 'r':
    case 'R':
      prec = 4;
      break;

    case 'x':
    case 'X':
    case 'p':
    case 'P':
      prec = 6;
      break;

    default:
      *sizeP = 0;
      return _("bad call to md_atof()");
    }

  t = atof_ieee (input_line_pointer, type, words);
  if (t)
    input_line_pointer = t;

  *sizeP = prec * sizeof (LITTLENUM_TYPE);

  for (wordP = words; prec--;)
    {
      md_number_to_chars (litP, (valueT) (*wordP++), sizeof (LITTLENUM_TYPE));
      litP += sizeof (LITTLENUM_TYPE);
    }

  return NULL;
}

/* GAS will call this function when a symbol table lookup fails, before it
   creates a new symbol.  Typically this would be used to supply symbols whose
   name or value changes dynamically, possibly in a context sensitive way.
   Predefined symbols with fixed values, such as register names or condition
   codes, are typically entered directly into the symbol table when `md_begin
   is called.  One argument is passed, a `char * for the symbol.  */

symbolS *
md_undefined_symbol (char *name ATTRIBUTE_UNUSED)
{
  return NULL;
}

/* GAS will call this function with one argument, an `expressionS pointer,
   for any expression that can not be recognized.  When the function is called,
   `input_line_pointer will point to the start of the expression.  */

void
md_operand (expressionS *exp ATTRIBUTE_UNUSED)
{
  return;
}

/* GAS will call this function for each section at the end of the assembly, to
   permit the CPU backend to adjust the alignment of a section.  The function
   must take two arguments, a `segT for the section and a `valueT for the
   size of the section, and return a `valueT for the rounded size.  */

valueT
md_section_align (segT sec, valueT size)
{
  int align = bfd_get_section_alignment (stdoutput, sec);
  return ((size + (1 << align) - 1) & (-1 << align));
  //return size;
}

/* The location from which a PC relative jump should be calculated,
   given a PC relative reloc.  */

/* MD_PCREL_FROM_SECTION (fixp, section)
   If you define this macro, it should return the position from which the PC
   relative adjustment for a PC relative fixup should be made.  On many
   processors, the base of a PC relative instruction is the next instruction,
   so this macro would return the length of an instruction, plus the address
   of the PC relative fixup.  The latter can be calculated as
   `fixp->fx_where + fixp->fx_frag->fr_address.  */

/* This is the default value of MD_PCREL_FROM_SECTION.  The difference is
   that `md_pcrel_from does not take a section argument.  */

long
md_pcrel_from (fixS *fixP) //segT sec)
{
  //if (fixP->fx_addsy != (symbolS *) NULL
  //    && (! S_IS_DEFINED (fixP->fx_addsy)
  //        || S_GET_SEGMENT (fixP->fx_addsy) != sec))
  //  {
      /* The symbol is undefined (or is defined but not in this section).
         Let the linker figure it out.  */
  //    return 0;
  //  }

  /* Return the address of the delay slot.  */
  return fixP->fx_frag->fr_address + fixP->fx_where + fixP->fx_size;
}

/* GAS will call this for each fixup that passes the TC_VALIDATE_FIX test
   when `linkrelax is not set.  It should store the correct value in the
   object file.  `fixS *fixP is the fixup `md_apply_fix is operating on.
   `valueT *valP is the value to store into the object files, or at least
   is the generic code's best guess.  Specifically, *valP is the value of
   the fixup symbol, perhaps modified by MD_APPLY_SYM_VALUE, plus
   `fixP->fx_offset (symbol addend), less MD_PCREL_FROM_SECTION for
   pc-relative fixups.  `segT seg is the section the fix is in.
   `fixup_segment performs a generic overflow check on *valP after
   `md_apply_fix returns.  If the overflow check is relevant for the target
   machine, then `md_apply_fix should modify *valP, typically to the
   value stored in the object file.  */

void
md_apply_fix (fixS *fixP, valueT *valP, segT seg ATTRIBUTE_UNUSED)
{
  char *buf = fixP->fx_frag->fr_literal + fixP->fx_where;
  valueT val = *valP;
  valueT insn = 0;

  assert (fixP->fx_r_type < BFD_RELOC_UNUSED);

#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG: md_apply_fix(%s, from line(%u), with val(%lx))\n",
          bfd_get_reloc_code_name (fixP->fx_r_type), fixP->fx_line, *valP);
  printf ("COFFEE_DEBUG:\tfx_size=%u, fx_pcrel=%u, fx_done=%u, fx_addsy=%i, fx_offset=%li\n",
          fixP->fx_size, fixP->fx_pcrel, fixP->fx_done, (fixP->fx_addsy ? 1 : 0), fixP->fx_offset);
#endif

  if (fixP->fx_subsy != (symbolS *) NULL)
    {
      /* We can't actually support subtracting a symbol.  */
      as_bad_where (fixP->fx_file, fixP->fx_line, _("expression too complex"));
    }

  if (fixP->fx_addsy != NULL)
    {
      if (fixP->fx_pcrel)
        {
          /* Hack around bfd_install_relocation brain damage.  */
          val += fixP->fx_frag->fr_address + fixP->fx_where;

          switch (fixP->fx_r_type)
            {
            case BFD_RELOC_32:
              fixP->fx_r_type = BFD_RELOC_32_PCREL;
              break;

            case BFD_RELOC_COFFEE_BRANCH:
            case BFD_RELOC_COFFEE_JUMP:
            case BFD_RELOC_32_PCREL:
              break;

            default:
              as_bad_where (fixP->fx_file, fixP->fx_line,
                            _("expression too complex"));
              break;
            }
        }
    }

  /* Remember value for emit_reloc.  */
  fixP->fx_addnumber = val;

  /* No need to go on if it's a relocation against an external symbol.  */
  if (fixP->fx_addsy == NULL && !fixP->fx_pcrel)
    {
      fixP->fx_done = 1;

      /* If this is a data relocation, just output VAL.  */
      if (fixP->fx_r_type == BFD_RELOC_8)
        {
          md_number_to_chars (buf, val, 1);
        }
      else if (fixP->fx_r_type == BFD_RELOC_16)
        {
          md_number_to_chars (buf, val, 2);
        }
      else if (fixP->fx_r_type == BFD_RELOC_32)
        {
          md_number_to_chars (buf, val, 4);
        }
      else if (fixP->fx_r_type == BFD_RELOC_64)
        {
          md_number_to_chars (buf, val, 8);
        }
      else
        {
          /* It's a relocation against an instruction.  */

          insn = bfd_getb32 (buf);

          switch (fixP->fx_r_type)
            {
            case BFD_RELOC_LO16:
              //if (!in_bitfield_range(val,0xffff))
              //  as_bad_where (fixP->fx_file, fixP->fx_line,
              //                _("relocation overflow"));
              insn |= ((val >> 15) & 0x1) << 9;
              insn |= (val & 0x7fff) << 10;
              break;

            case BFD_RELOC_HI16:
              //if (!in_bitfield_range(val,0xffff))
              //  as_bad_where (fixP->fx_file, fixP->fx_line,
              //                _("relocation overflow"));
              val >>= 16;
              insn |= ((val >> 15) & 0x1) << 9;
              insn |= (val & 0x7fff) << 10;
              break;

            case BFD_RELOC_COFFEE_BRANCH:
              if (!in_signed_range(val,0x1fffff))
                as_bad_where (fixP->fx_file, fixP->fx_line,
                              _("relocation overflow"));
              val >>= 1;
              insn |= (val & 0x003fffff);
              break;

            case BFD_RELOC_COFFEE_JUMP:
              if (!in_signed_range(val,0x7fffff))
                as_bad_where (fixP->fx_file, fixP->fx_line,
                              _("relocation overflow"));
              val >>= 1;
              insn |= (val & 0x01ffffff);
              break;

            case BFD_RELOC_NONE:
            default:
              as_bad_where (fixP->fx_file, fixP->fx_line,
                            _("reloc %d not supported by object file format"),
                            fixP->fx_r_type);
              break;
            }

          bfd_putb32 (insn, buf);
        }
    }

#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG:\tinsn=0x%lX, fx_done=%i\n", insn, fixP->fx_done);
#endif
}

/* GAS will call this to generate a reloc.  GAS will pass the resulting
   reloc to `bfd_install_relocation'.  This currently works poorly, as
   `bfd_install_relocation often does the wrong thing, and instances of
   `tc_gen_reloc have been written to work around the problems,
   which in turns makes it difficult to fix `bfd_install_relocation.*/

arelent *
tc_gen_reloc (asection *section ATTRIBUTE_UNUSED, fixS *fixP)
{
#if COFFEE_DEBUG
  printf ("COFFEE_DEBUG: tc_gen_reloc(%s, from line (%u))\n",
          bfd_get_reloc_code_name (fixP->fx_r_type), fixP->fx_line);
#endif

  arelent *reloc;
  reloc = (arelent *) xmalloc (sizeof (arelent));
  reloc->sym_ptr_ptr = (asymbol **) xmalloc (sizeof (asymbol *));
  *reloc->sym_ptr_ptr = symbol_get_bfdsym (fixP->fx_addsy);
  reloc->address = fixP->fx_frag->fr_address + fixP->fx_where;
  reloc->addend = fixP->fx_addnumber;
  reloc->howto = bfd_reloc_type_lookup (stdoutput, fixP->fx_r_type);
  if (!reloc->howto)
    {
      as_bad_where (fixP->fx_file, fixP->fx_line,
                    _("internal error: can't export reloc type %d (`%s')"),
                    fixP->fx_r_type, bfd_get_reloc_code_name (fixP->fx_r_type));
      return NULL;
    }
  return reloc;
}

/* This function returns an estimate of the size of a rs_machine_dependent
   frag before any relaxing is done.  It may also create any necessary
   relocations.  */

int
md_estimate_size_before_relax (fragS *fragP ATTRIBUTE_UNUSED,
                               segT segment ATTRIBUTE_UNUSED)
{
  as_fatal (_("md_estimate_size_before_relax()"));
  return 0;
}

/* GAS will call this for each rs_machine_dependent fragment.
   The instruction is completed using the data from the relaxation pass.
   It may also create any necessary relocations.  */

void
md_convert_frag (bfd *abfd ATTRIBUTE_UNUSED,
                 segT segment ATTRIBUTE_UNUSED,
                 fragS *fragP ATTRIBUTE_UNUSED)
{
  as_fatal (_("md_convert_frag()"));
}
