/* coffee-dis.c -- Disassemble COFFEE instructions
   Copyright 2005, 2006 Free Software Foundation, Inc.

   This file is part of GDB, GAS, and the GNU binutils.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street - Fifth Floor, Boston, MA 02110-1301, USA.  */

#include "sysdep.h"
#include "dis-asm.h"
#include "opcode/coffee.h"

/* This file provides a disassembler function which uses
   the disassembler interface defined in dis-asm.h.   */

/* Print one instruction from MEMADDR on INFO->STREAM.  */
int
print_insn_coffee (bfd_vma memaddr, disassemble_info *info)
{
  bfd_byte buffer[4];
  int status;
  unsigned int insn;
  const struct coffee_opcode *opcode;
  const char *arg;
  long val;

  status = (*info->read_memory_func) (memaddr, buffer, 4, info);
  if (status != 0)
    {
      (*info->memory_error_func) (status, memaddr, info);
      return -1;
    }

  insn = bfd_getb32 (buffer);

  for (opcode = coffee_opcodes; opcode->name; opcode++)
    {
      if ((insn & OPC_MASK) == (opcode->opcode & OPC_MASK))
        {
          /* There are two variations of each shift instruction
             so we need to select the correct one.  */
          if ((opcode->flags & F_SHIFT) && !(insn & SM(1)))
            ++opcode;

          (*info->fprintf_func) (info->stream, opcode->name);

          for (arg = opcode->args; *arg != '\0'; ++arg)
            {

              if (*arg == ',')
                {
                  (*info->fprintf_func) (info->stream, ",");
                  ++arg;
                }
              else
                {
                  (*info->fprintf_func) (info->stream, " ");
                }

              switch (*arg)
                {
                case 'r':
                  switch (*++arg)
                    {
                    case 'D':
                      val = ((insn >> 0) & 0x1f);
                      (*info->fprintf_func) (info->stream, "r%lu", val);
                      continue;
                    case 'A':
                      val = ((insn >> 5) & 0x1f);
                      (*info->fprintf_func) (info->stream, "r%lu", val);
                      continue;
                    case 'B':
                      val = ((insn >> 10) & 0x1f);
                      (*info->fprintf_func) (info->stream, "r%lu", val);
                      continue;
                    case 'C':
                      val = ((insn >> 22) & 0x3);
                      (*info->fprintf_func) (info->stream, "c%lu", val);
                      continue;
                    case 'P':
                      val = ((insn >> 12) & 0x1f);
                      (*info->fprintf_func) (info->stream, "cr%lu", val);
                      continue;
                    default:
                      (*info->fprintf_func) (info->stream, "%c", *arg);
                      continue;
                    }

                case 'c':
                  switch (*++arg)
                    {
                    case 'I':
                      val = ((((insn >> 10) & 0x7fff) ^ 0x4000) - 0x4000);
                      (*info->fprintf_func) (info->stream, "%li", val);
                      continue;
                    case 'i':
                      val = ((((insn >> 10) & 0x1ff) ^ 0x100) - 0x100);
                      (*info->fprintf_func) (info->stream, "%li", val);
                      continue;
                    case 'U':
                      val = ((insn >> 10) & 0x7fff);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'u':
                      val = ((insn >> 10) & 0x1ff);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'C':
                      val = ((insn & 0x1f) << 12);
                      val |= ((insn >> 10) & 0xfff);
                      val = (val ^ 0x10000) - 0x10000;
                      (*info->fprintf_func) (info->stream, "%li", val);
                      continue;
                    case 'S':
                      val = ((insn & 0x1f) << 10);
                      val |= ((insn >> 15) & 0x3ff);
                      val = (val ^ 0x4000) - 0x4000;
                      (*info->fprintf_func) (info->stream, "%li", val);
                      continue;
                    case 'K':
                      val = ((insn >> 10) & 0x3f);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'N':
                      val = ((insn >> 10) & 0x3);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'n':
                      val = ((insn >> 10) & 0x1);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'B':
                      val = ((insn >> 15) & 0x3f);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'b':
                      val = ((insn >> 10) & 0x1f);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'P':
                      val = ((insn >> 24) & 0x3);
                      (*info->fprintf_func) (info->stream, "%lu", val);
                      continue;
                    case 'O':
                      val = ((insn >> 0) & 0xffffff);
                      (*info->fprintf_func) (info->stream, "%#lx", val);
                      continue;
                    case 'X':
                      val = ((insn >> 9) & 0x1) << 15;
                      val |= ((insn >> 10) & 0x7fff);
                      (*info->fprintf_func) (info->stream, "%#lx", val);
                      continue;
                    default:
                      (*info->fprintf_func) (info->stream, "%c", *arg);
                      continue;
                    }

                case 'b':
                  switch (*++arg)
                    {
                    case 'B':
                      val = (insn & 0x1fffff);
                      val ^= 0x100000;
                      val -= 0x100000;
                      (*info->fprintf_func) (info->stream, "%ld", val);
                      continue;
                    case 'J':
                      val = (insn & 0x1ffffff);
                      val ^= 0x1000000;
                      val -= 0x1000000;
                      (*info->fprintf_func) (info->stream, "%ld", val);
                      continue;
                    default:
                      (*info->fprintf_func) (info->stream, "%c", *arg);
                      continue;
                    }

                default:
                  (*info->fprintf_func) (info->stream, "%c", *arg);
                  continue;
                }
            }
          /* We have found and printed an instruction; return.  */
          return 4;
        }
    }

  /* We could not find a match.  */
  (*info->fprintf_func) (info->stream, ".long %#x", insn);

  return 4;
}
