/* coffee-opc.c -- Opcode list for COFFEE
   Copyright 2005, 2006 Free Software Foundation, Inc.

   This file is part of GDB, GAS, and the GNU binutils.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street - Fifth Floor, Boston, MA 02110-1301, USA.  */

#include "opcode/coffee.h"

/* - The match component is a mask saying which bits must match a
     particular opcode in order for an instruction to be an instance
     of that opcode.

   - The args component is a string containing characters symbolically
     matching the operands of an instruction.  Used for both assembly
     and disassembly.

     Operand-matching characters:
     ,  Itself.  The character appears in the assembly code verbatim.
     rD (dreg)   Destination register operand in bits <4:0>.
     rA (sreg1)  Source register in bits <9:5>.
     rB (sreg2)  Source register in bits <14:10>.
     rC (creg)   Condition register in bits <23:22>.
     rP (cp_reg) Coprocessor register in bits <16:12>.
     cI (imm)    15-bit signed immediate constant in bits <24:10>.
     ci (imm)    9-bit signed immediate constant in bits <18:10>.
     cU (imm)    15-bit unsigned immediate constant in bits <24:10>.
     cu (imm)    9-bit unsigned immediate constant in bits <18:10>.
     cC (imm)    17-bit signed immediate divided in bits <21:10> and <4:0>.
     cS (imm)    15-bit signed immediate divided in bits <24:15> and <4:0>.
     cK (imm)    Small constant of 0..32 in bits <15:10>.
     cN (imm)    Small constant of 0..3 in bits <11:10>.
     cn (imm)    Small constant of 0 or 1 in bit <10>.
     cB (imm1)   Bitfield length (0-32) in bits <20:15>.
     cb (imm2)   Bitfield position (0-31) in bits <14:10>.
     cP (imm1)   Coprocessor number in bits <25:24>.
     cO (imm2)   Coprocessor instruction in bits <23:0>.
     cX (imm)    16-bit signed or unsigned immediate in bits <24:10> except MSB in <9>.
     bB (imm)    22-bit branch address in bits <21:0>.
     bJ (imm)    25-bit jump address in bits <24:0>.  */

const struct coffee_opcode coffee_opcodes[] =
{
  {"add",	OP(0x01),		"rD,rA,rB",	F_CEX},
  {"addi",	OP(0x2D),		"rD,rA,cI",	F_CEX},
  {"addiu",	OP(0x28),		"rD,rA,cU",	F_CEX},
  {"addu",	OP(0x00),		"rD,rA,rB",	F_CEX},
  {"and",	OP(0x02),		"rD,rA,rB",	F_CEX},
  {"andi",	OP(0x29),		"rD,rA,cU",	F_CEX},
  {"bc",	OP(0x20)|CEX(1),	"rC,bB",	F_BRANCH},
  {"begt",	OP(0x21)|CEX(1),	"rC,bB",	F_BRANCH},
  {"belt",	OP(0x22)|CEX(1),	"rC,bB",	F_BRANCH},
  {"beq",	OP(0x23)|CEX(1),	"rC,bB",	F_BRANCH},
  {"bgt",	OP(0x24)|CEX(1),	"rC,bB",	F_BRANCH},
  {"blt",	OP(0x25)|CEX(1),	"rC,bB",	F_BRANCH},
  {"bne",	OP(0x26)|CEX(1),	"rC,bB",	F_BRANCH},
  {"bnc",	OP(0x27)|CEX(1),	"rC,bB",	F_BRANCH},
  {"chrs",	OP(0x33),		"cN",		0},
  {"cmp",	OP(0x19),		"rC,rA,rB",	0},
  {"cmpi",	OP(0x37),		"rC,rA,cC",	0},
  {"conb",	OP(0x03),		"rD,rA,rB",	F_CEX},
  {"conh",	OP(0x04),		"rD,rB,rA",	F_CEX},
  {"cop",	OP(0x3C),		"cP,cO",	0},
  {"di",	OP(0x15),		"",		0},
  {"ei",	OP(0x16),		"",		0},
  {"exb",	OP(0x30),		"rD,rA,cN",	F_CEX},
  {"exbf",	OP(0x1A),		"rD,rA,rB",	F_CEX},
  {"exbfi",	OP(0x3D),		"rD,rA,cB,cb",	0},
  {"exh",	OP(0x31),		"rD,rA,cn",	F_CEX},
  {"jal",	OP(0x39),		"bJ",		F_BRANCH},
  {"jalr",	OP(0x35)|RD(0x1F),	"rA",		F_CEX},
  {"jmp",	OP(0x38),		"bJ",		F_BRANCH},
  {"jmpr",	OP(0x1B),		"rA",		F_CEX|F_BRANCH},
  {"ld",	OP(0x32),		"rD,rA,cI",	F_CEX},
  {"lli",	OP(0x3E),		"rD,cX",	0},
  {"lui",	OP(0x3F),		"rD,cX",	0},
  {"mov",	OP(0x13),		"rD,rA",	F_CEX},
  {"movfc",	OP(0x2C),		"cN,rD,rP",	F_CEX},
  {"movtc",	OP(0x36),		"cN,rP,rA",	F_CEX},
  {"mulhi",	OP(0x1D),		"rD",		F_CEX},
  {"muli",	OP(0x2E),		"rD,rA,cI",	F_CEX},
  {"muls",	OP(0x05),		"rD,rA,rB",	F_CEX},
  {"muls_16",	OP(0x08),		"rD,rA,rB",	F_CEX},
  {"mulu",	OP(0x06),		"rD,rA,rB",	F_CEX},
  {"mulu_16",	OP(0x09),		"rD,rA,rB",	F_CEX},
  {"mulus",	OP(0x07),		"rD,rA,rB",	F_CEX},
  {"mulus_16",	OP(0x0A),		"rD,rA,rB",	F_CEX},
  {"nop",	NOP,			"",		0},
  {"not",	OP(0x14),		"rD,rA",	F_CEX},
  {"or",	OP(0x0B),		"rD,rA,rB",	F_CEX},
  {"ori",	OP(0x2A),		"rD,rA,cU",	F_CEX},
  {"rcon",	OP(0x1E),		"rA",		0},
  {"reti",	OP(0x17),		"",		0},
  {"retu",	OP(0x1F)|RA(0x1F),	"",		0},
  {"scall",	OP(0x3B)|RD(0x1F),	"",		F_CEX},
  {"scon",	OP(0x1C),		"rD",		0},
  {"sext",	OP(0x0C),		"rD,rA,rB",	F_CEX},
  {"sexti",	OP(0x2B),		"rD,rA,cb",	F_CEX},
  {"sll",	OP(0x0D)|SM(1),		"rD,rA,rB",	F_CEX|F_SHIFT},
  {"slli",	OP(0x0D),		"rD,rA,cK",	F_CEX|F_SHIFT},
  {"sra",	OP(0x0E)|SM(1),		"rD,rA,rB",	F_CEX|F_SHIFT},
  {"srai",	OP(0x0E),		"rD,rA,cK",	F_CEX|F_SHIFT},
  {"srl",	OP(0x0F)|SM(1),		"rD,rA,rB",	F_CEX|F_SHIFT},
  {"srli",	OP(0x0F),		"rD,rA,cK",	F_CEX|F_SHIFT},
  {"st",	OP(0x34),		"rB,rA,cS",	F_CEX},
  {"sub",	OP(0x10),		"rD,rA,rB",	F_CEX},
  {"subu",	OP(0x11),		"rD,rA,rB",	F_CEX},
  {"swm",	OP(0x2F),		"cK",		0},
  {"trap",	OP(0x18),		"cb",		F_CEX},
  {"xor",	OP(0x12),		"rD,rA,rB",	F_CEX},
  {0, 0, 0, 0}
};

/*   Operand-matching characters for Milk:
     ,  Itself.  The character appears in the assembly code verbatim.
     mD (dr)     Milk result register in bits <10:6>.
     mA (sr1)    Milk operand register in bits <15:11>.
     mB (sr2)    Milk operand register in bits <20:16>.  */
const struct coffee_opcode milk_opcodes[] =
{
  {"fadd0",	0xF0000000,	"mD,mA,mB", 	F_MILK},
  {"fadd1",	0xF1000000,	"mD,mA,mB", 	F_MILK},
  {"fadd2",	0xF2000000,	"mD,mA,mB", 	F_MILK},
  {"fadd3",	0xF3000000,	"mD,mA,mB", 	F_MILK},
  {"fsub0",	0xF0000001,	"mD,mA,mB", 	F_MILK},
  {"fsub1",	0xF1000001,	"mD,mA,mB", 	F_MILK},
  {"fsub2",	0xF2000001,	"mD,mA,mB", 	F_MILK},
  {"fsub3",	0xF3000001,	"mD,mA,mB", 	F_MILK},
  {"fmul0",	0xF0000002,	"mD,mA,mB", 	F_MILK},
  {"fmul1",	0xF1000002,	"mD,mA,mB", 	F_MILK},
  {"fmul2",	0xF2000002,	"mD,mA,mB", 	F_MILK},
  {"fmul3",	0xF3000002,	"mD,mA,mB", 	F_MILK},
  {"fdiv0",	0xF0000003,	"mD,mA,mB", 	F_MILK},
  {"fdiv1",	0xF1000003,	"mD,mA,mB", 	F_MILK},
  {"fdiv2",	0xF2000003,	"mD,mA,mB", 	F_MILK},
  {"fdiv3",	0xF3000003,	"mD,mA,mB", 	F_MILK},
  {"fsqrt0",	0xF0000004,	"mD,mA", 	F_MILK},
  {"fsqrt1",	0xF1000004,	"mD,mA", 	F_MILK},
  {"fsqrt2",	0xF2000004,	"mD,mA", 	F_MILK},
  {"fsqrt3",	0xF3000004,	"mD,mA", 	F_MILK},
  {"fabs0",	0xF0000005,	"mD,mA", 	F_MILK},
  {"fabs1",	0xF1000005,	"mD,mA", 	F_MILK},
  {"fabs2",	0xF2000005,	"mD,mA", 	F_MILK},
  {"fabs3",	0xF3000005,	"mD,mA", 	F_MILK},
  {"fmov0",	0xF0000006,	"mD,mA", 	F_MILK},
  {"fmov1",	0xF1000006,	"mD,mA", 	F_MILK},
  {"fmov2",	0xF2000006,	"mD,mA", 	F_MILK},
  {"fmov3",	0xF3000006,	"mD,mA", 	F_MILK},
  {"fneg0",	0xF0000007,	"mD,mA", 	F_MILK},
  {"fneg1",	0xF1000007,	"mD,mA", 	F_MILK},
  {"fneg2",	0xF2000007,	"mD,mA", 	F_MILK},
  {"fneg3",	0xF3000007,	"mD,mA", 	F_MILK},
  {"fnop0",	0xF0000008,	"", 		F_MILK},
  {"fnop1",	0xF1000008,	"", 		F_MILK},
  {"fnop2",	0xF2000008,	"", 		F_MILK},
  {"fnop3",	0xF3000008,	"", 		F_MILK},
  {"fcvt.s0",	0xF0000020,	"mD,mA", 	F_MILK},
  {"fcvt.s1",	0xF1000020,	"mD,mA", 	F_MILK},
  {"fcvt.s2",	0xF2000020,	"mD,mA", 	F_MILK},
  {"fcvt.s3",	0xF3000020,	"mD,mA", 	F_MILK},
  {"fcvt.w0",	0xF0000024,	"mD,mA", 	F_MILK},
  {"fcvt.w1",	0xF1000024,	"mD,mA", 	F_MILK},
  {"fcvt.w2",	0xF2000024,	"mD,mA", 	F_MILK},
  {"fcvt.w3",	0xF3000024,	"mD,mA", 	F_MILK},
  {"fc.f0",	0xF0000030,	"mD,mA,mB", 	F_MILK},
  {"fc.f1",	0xF1000030,	"mD,mA,mB", 	F_MILK},
  {"fc.f2",	0xF2000030,	"mD,mA,mB", 	F_MILK},
  {"fc.f3",	0xF3000030,	"mD,mA,mB", 	F_MILK},
  {"fc.un0",	0xF0000031,	"mD,mA,mB", 	F_MILK},
  {"fc.un1",	0xF1000031,	"mD,mA,mB", 	F_MILK},
  {"fc.un2",	0xF2000031,	"mD,mA,mB", 	F_MILK},
  {"fc.un3",	0xF3000031,	"mD,mA,mB", 	F_MILK},
  {"fc.eq0",	0xF0000032,	"mD,mA,mB", 	F_MILK},
  {"fc.eq1",	0xF1000032,	"mD,mA,mB", 	F_MILK},
  {"fc.eq2",	0xF2000032,	"mD,mA,mB", 	F_MILK},
  {"fc.eq3",	0xF3000032,	"mD,mA,mB", 	F_MILK},
  {"fc.ueq0",	0xF0000033,	"mD,mA,mB", 	F_MILK},
  {"fc.ueq1",	0xF1000033,	"mD,mA,mB", 	F_MILK},
  {"fc.ueq2",	0xF2000033,	"mD,mA,mB", 	F_MILK},
  {"fc.ueq3",	0xF3000033,	"mD,mA,mB", 	F_MILK},
  {"fc.olt0",	0xF0000034,	"mD,mA,mB", 	F_MILK},
  {"fc.olt1",	0xF1000034,	"mD,mA,mB", 	F_MILK},
  {"fc.olt2",	0xF2000034,	"mD,mA,mB", 	F_MILK},
  {"fc.olt3",	0xF3000034,	"mD,mA,mB", 	F_MILK},
  {"fc.ult0",	0xF0000035,	"mD,mA,mB", 	F_MILK},
  {"fc.ult1",	0xF1000035,	"mD,mA,mB", 	F_MILK},
  {"fc.ult2",	0xF2000035,	"mD,mA,mB", 	F_MILK},
  {"fc.ult3",	0xF3000035,	"mD,mA,mB", 	F_MILK},
  {"fc.ole0",	0xF0000036,	"mD,mA,mB", 	F_MILK},
  {"fc.ole1",	0xF1000036,	"mD,mA,mB", 	F_MILK},
  {"fc.ole2",	0xF2000036,	"mD,mA,mB", 	F_MILK},
  {"fc.ole3",	0xF3000036,	"mD,mA,mB", 	F_MILK},
  {"fc.ule0",	0xF0000037,	"mD,mA,mB", 	F_MILK},
  {"fc.ule1",	0xF1000037,	"mD,mA,mB", 	F_MILK},
  {"fc.ule2",	0xF2000037,	"mD,mA,mB", 	F_MILK},
  {"fc.ule3",	0xF3000037,	"mD,mA,mB", 	F_MILK},
  {"fc.sf0",	0xF0000038,	"mD,mA,mB", 	F_MILK},
  {"fc.sf1",	0xF1000038,	"mD,mA,mB", 	F_MILK},
  {"fc.sf2",	0xF2000038,	"mD,mA,mB", 	F_MILK},
  {"fc.sf3",	0xF3000038,	"mD,mA,mB", 	F_MILK},
  {"fc.ngle0",	0xF0000039,	"mD,mA,mB", 	F_MILK},
  {"fc.ngle1",	0xF1000039,	"mD,mA,mB", 	F_MILK},
  {"fc.ngle2",	0xF2000039,	"mD,mA,mB", 	F_MILK},
  {"fc.ngle3",	0xF3000039,	"mD,mA,mB", 	F_MILK},
  {"fc.seq0",	0xF000003A,	"mD,mA,mB", 	F_MILK},
  {"fc.seq1",	0xF100003A,	"mD,mA,mB", 	F_MILK},
  {"fc.seq2",	0xF200003A,	"mD,mA,mB", 	F_MILK},
  {"fc.seq3",	0xF300003A,	"mD,mA,mB", 	F_MILK},
  {"fc.ngl0",	0xF000003B,	"mD,mA,mB", 	F_MILK},
  {"fc.ngl1",	0xF100003B,	"mD,mA,mB", 	F_MILK},
  {"fc.ngl2",	0xF200003B,	"mD,mA,mB", 	F_MILK},
  {"fc.ngl3",	0xF300003B,	"mD,mA,mB", 	F_MILK},
  {"fc.lt0",	0xF000003C,	"mD,mA,mB", 	F_MILK},
  {"fc.lt1",	0xF100003C,	"mD,mA,mB", 	F_MILK},
  {"fc.lt2",	0xF200003C,	"mD,mA,mB", 	F_MILK},
  {"fc.lt3",	0xF300003C,	"mD,mA,mB", 	F_MILK},
  {"fc.nge0",	0xF000003D,	"mD,mA,mB", 	F_MILK},
  {"fc.nge1",	0xF100003D,	"mD,mA,mB", 	F_MILK},
  {"fc.nge2",	0xF200003D,	"mD,mA,mB", 	F_MILK},
  {"fc.nge3",	0xF300003D,	"mD,mA,mB", 	F_MILK},
  {0, 0, 0, 0}
};

const struct coffee_reg coffee_regs[] =
{
  {"r0", 0},
  {"r1", 1},
  {"r2", 2},
  {"r3", 3},
  {"r4", 4},
  {"r5", 5},
  {"r6", 6},
  {"r7", 7},
  {"r8", 8},
  {"r9", 9},
  {"r10", 10},
  {"r11", 11},
  {"r12", 12},
  {"r13", 13},
  {"r14", 14},
  {"r15", 15},
  {"r16", 16},
  {"r17", 17},
  {"r18", 18},
  {"r19", 19},
  {"r20", 20},
  {"r21", 21},
  {"r22", 22},
  {"r23", 23},
  {"r24", 24},
  {"r25", 25},
  {"r26", 26},
  {"r27", 27},
  {"r28", 28},
  {"r29", 29},
  {"r30", 30},
  {"r31", 31},
  {"c0", 0},
  {"c1", 1},
  {"c2", 2},
  {"c3", 3},
  {"c4", 4},
  {"c5", 5},
  {"c6", 6},
  {"c7", 7},
  {"SPSR", 30},
  {"LR", 31},
  {0, 0}
};

const struct coffee_reg milk_regs[] =
{
  {"cr0", 0},
  {"cr1", 1},
  {"cr2", 2},
  {"cr3", 3},
  {"cr4", 4},
  {"cr5", 5},
  {"cr6", 6},
  {"cr7", 7},
  {"cr8", 8},
  {"cr9", 9},
  {"cr10", 10},
  {"cr11", 11},
  {"cr12", 12},
  {"cr13", 13},
  {"cr14", 14},
  {"cr15", 15},
  {"cr16", 16},
  {"cr17", 17},
  {"cr18", 18},
  {"cr19", 19},
  {"cr20", 20},
  {"cr21", 21},
  {"cr22", 22},
  {"cr23", 23},
  {"cr24", 24},
  {"cr25", 25},
  {"cr26", 26},
  {"cr27", 27},
  {"cr28", 28},
  {"cr29", 29},
  {"cr30", 30},
  {"cr31", 31},
  {0, 0}
};

const struct coffee_condcode coffee_condcodes[] =
{
  {"c", 0},
  {"egt", 1},
  {"elt", 1},
  {"eq", 3},
  {"gt", 4},
  {"lt", 5},
  {"ne", 6},
  {"nc", 7},
  {0, 0}
};
