// C++ implementation of arithmetic instructions of COFFEE RISC core
// Marko Savolainen
// marko.k.savolainen@tut.fi
// Tampere University of Technology
// Institute of Digital and Computer Systems

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <cstdlib>
#include <cmath>
using namespace std;
#include "arith_oper.h"

//Function that performs 32*32bit = 64 bit unsigned multiplication
//See COFFEE specs for implementation details
unsigned long multiply_32bit(unsigned long operand1,unsigned long operand2, unsigned long& mresult_hi) {
  int carry = 0;
  unsigned long op1_lo = 0x0000ffff & operand1;
  unsigned long op2_lo = 0x0000ffff & operand2;
  unsigned long op1_hi = operand1 >> 16;
  unsigned long op2_hi = operand2 >> 16;
  unsigned long temp_res  = op1_lo * op2_lo;
  unsigned long temp_res2 = op1_lo * op2_hi;
  unsigned long temp_res3 = op1_hi * op2_lo;
  unsigned long temp_res4 = op1_hi * op2_hi; 
  unsigned long temp_res5 = (temp_res2 & 0x0000ffff) + (temp_res3 &  0x0000ffff) + (temp_res >> 16);
   
  if( (temp_res5 >> 16) != 0) {
    carry = true;
  
  } 
   else {
    carry = false;
  } 
  temp_res = (temp_res5 << 16) + (temp_res & 0x0000ffff); 
  mresult_hi = temp_res4 + (temp_res2 >> 16) + (temp_res3 >> 16) + carry;
  return temp_res;
}

// Function to convert signed number to unsigned format
unsigned long convert_sign_unsign( signed long operand){
  return abs(operand);
}


//Function to convert unsigned multiplication result to signed format
signed long mult_signed(int sign, unsigned long mult_result_lo, unsigned long mult_result_hi, signed long& mult_res_sign_hi) {    
  // positive result -> no changes need to be done
  if (sign == 0 ) {
    mult_res_sign_hi = mult_result_hi;
    return mult_result_lo;
  }  else { // conversion is needed when the result is negative
    if (mult_result_lo == 0) {//carry to upper part
      
      mult_res_sign_hi = (~mult_result_hi) + 1;
      return mult_result_lo;
    } else {
      mult_res_sign_hi = ~mult_result_hi;
      return (~mult_result_lo) + 1;
    }
  }
}


//Function add performs the adding operation 
long add(long operand1, long operand2, bool& overflow, bool& underflow, int& C,int& Z , int& N, bool& addr_oflow) {
  long temp = operand1 + operand2;
  //updating flags
  if ( temp == 0 ) {
    Z = 1;
  } else { 
    Z = 0;
  }
  if(temp < 0) {
    N = 1;
  }else{
    N = 0;
  }
  //finding carry flag
  bool carry = false;
  long temp1 = 0;
  long temp2 = 0;
  long temp3 = 1;
  for(int i=0; i<32; i++) {
    temp1 = operand1 & temp3;
    temp2 = operand2 & temp3;  
    if((temp1 | temp2)!=0){
      if((temp1 & temp2)!=0) {
	carry = true;
      }else if(carry) {
	carry = true;
      }else{
	carry = false;
      }
    }else{
      carry=false;
    }
    temp3 = 2*temp3;
      
  }
 
  if (carry) { 
    C = 1;
  } else {
    C = 0;
  }

  // updating overflow / underflow
  if( ( operand1 > 0 and operand2 > 0) and temp < 0 ) {

    overflow = 1;
  } else {
    overflow = 0;
  }

  if( ( operand1 < 0 and operand2 < 0) and temp >= 0 ) {

    underflow = 1;
  } else {
    underflow = 0;
  }
  if((operand1 >= 0 and operand2<0 and temp <0) or (carry and operand1 < 0 and operand2 > 0)) {
    addr_oflow = true;
  }else{
    addr_oflow = false;
  }
    
  return temp;
}


//Function sub performs the substract operation 
long sub(long operand1, long operand2, bool& overflow, bool& underflow, int& C,int& Z , int& N, bool cmp_opr) {
  long temp = operand1 - operand2;
  long inv_op2 = (~operand2) +1;
  //updating flags
  if ( temp == 0 ) {
    Z = 1;
  } else { 
    Z = 0;
  }
  if (cmp_opr) {
    if(operand1<=0 and operand2 >0) {
      N = 1;
    }else if(operand1>=0 and operand2 <0){
      N = 0;
    }else if(temp<0) {
      N = 1;
    }else{
      N = 0;
    }
  }else{
    if(temp<0) {
      N = 1;
    }else{
      N = 0;
    }
  }

  //finding carry flag
  bool carry = false;
  long temp1 = 0;
  long temp2 = 0;
  long temp3 = 1;
  for(int i=0; i<32; i++) {
    temp1 = operand1 & temp3;
    temp2 = inv_op2 & temp3;  
    if((temp1 | temp2)!=0){
      if((temp1 & temp2)!=0) {
	carry = true;
      }else if(carry) {
	carry = true;
      }else{
	carry = false;
      }
    }else{
      carry=false;
    }
    temp3 = 2*temp3;
  }
 
  if (carry or operand2 == 0) { 
    C = 1;
  } else {
    C = 0;
  }

  // updating overflow / underflow
  if( ( operand1 >= 0 and operand2 < 0) and temp < 0 ) {

    overflow = 1;
  } else {
    overflow = 0;
  }

  if( ( operand1 < 0 and operand2 >= 0) and temp > 0 ) {

    underflow = 1;
  } else {
    underflow = 0;
  }
    
  return temp;
}

//Function and performs bitwise and operation
unsigned long and_f(unsigned long operand1, unsigned long operand2) {
  return (operand1 & operand2);
}    

//Function or performs bitwise or operation
unsigned long or_f(unsigned long operand1, unsigned long operand2) {
  return (operand1 | operand2);
}

//Function xor performs bitwise xor operation
unsigned long xor_f(unsigned long operand1, unsigned long operand2) {
  return (operand1 ^ operand2);
}


//Function not performs bitwise not operation
unsigned long not_f(unsigned long operand) {
  return ( ~operand );
}

//Function conb performs concatenate operation
unsigned long conb(unsigned long operand1, unsigned long operand2) {
  // moving bytes to right places
  unsigned temp1 = (operand1 & 0x000ff);
  unsigned temp2 = (operand2 & 0x000ff);
  temp1 = (temp1 << 8);

  return ( temp1 | temp2);
}

//Function conh performs concatenate operation
unsigned long conh(unsigned long operand1, unsigned long operand2) {
  // moving bytes to right places
  unsigned temp1 = (operand1 & 0x0000ffff);
  unsigned temp2 = (operand2 & 0x0000ffff);
  temp1 = (temp1 << 16);

  return ( temp1 | temp2);
}

//Function exb performs a bitfield extraction operation
unsigned long exb(unsigned long operand1, unsigned long operand2) {
  // Selecting right bitfield to extract
  unsigned long temp = operand1;
  switch (operand2) {
    case 0:
      temp = temp & 0x000ff;
      return temp;
    case 1:
      temp = temp & 0x00ff00;
      temp = (temp >> 8);
      return temp;
    case 2:
      temp = temp & 0x0ff0000;
      temp = (temp >> 16);
      return temp;
    case 3:
      temp = temp & 0xff000000;
      temp = (temp >> 24);
      return temp;
    default:
      return 0;
  }
}


//Function exbfi performs a bitfield extraction operation
unsigned long exbf(unsigned long operand1, unsigned long operand2) {
  
  unsigned int position = 0x001f & operand2;
  unsigned int length = (0x07e0 & operand2) >> 5; 
  unsigned long temp = 0;
  unsigned long temp2 = 0;
  
  temp = operand1 >> position;
  if(length + position < 32) {
    if(length == 0) {
      temp2 =0;
    }else if (length == 1) {
      temp2 = 1;
    } else{
      temp2 = 1;
      for ( unsigned int i = 0; i < length -1; i++) {
        temp2 = (temp2 << 1) +1;
      }
    }
    temp = temp & temp2;
    return temp;
  }else{
    return temp;
  }
    
}  

//Function exh performs a halfword extraction operation
unsigned long exh(unsigned long operand1, unsigned int select) {
   
  unsigned long temp = 0;
 
  if( select == 0) {
    temp = 0x00ffff & operand1;
  } else {
    temp = 0x00ffff & (operand1 >> 16);
  }

  return temp;
}     


//Function that performs power calculation(not an instruction)
unsigned long power(unsigned long operand1, unsigned long powe) {
  unsigned long temp = 0;
  if (powe == 0) {
    temp = 1;
  } else {
    temp = operand1;
    for (unsigned long i = 0; i < powe - 1; i++) {
      temp = temp * operand1;
    }
  }
  return temp;
}
    


//Function sext performs a sign extension operation
unsigned long sext(unsigned long operand1, unsigned int position) {
   
  unsigned long temp = 0;
  unsigned int sign = 0;
  // checking the value of sign bit
  temp = temp + power(2,position);
  if( (temp & operand1) > 0) {
    sign = 1;
  }else{
    sign = 0;
  }
  
  temp = 0;
  for( unsigned int i = position; i < 32; i++) {
    temp = temp + power(2,i);
  }
  //positive number
  if (sign == 0) {
    temp = operand1 & (~temp);
  } else {  //negative number
    temp = (operand1 | temp);
  }
      
  return temp;
}

//Function sll performs shift left logical operation
unsigned long sll(unsigned long operand1, unsigned long operand2, int& Z, int& N , int& C) {
  if (operand2 > 31) {
    operand2 = 32;
  }
  unsigned long temp = operand1 << operand2;
  if(operand2 == 32) {
    temp = 0;
  }

  if(temp == 0) {
     Z=1;
  }else{
     Z=0;
  }
   if( (temp & 0x80000000) != 0) {
     N=1;
  }else{
     N=0;
  }
  C=0;
  if(operand2 != 0) {
    if ( ((operand1<<(operand2 -1)) & 0x80000000) != 0) { 
      C = 1;
    }
  } 
  return temp;
}

//Function srl performs shift right logical operation
unsigned long srl(unsigned long operand1, unsigned long operand2) {
  if(operand2 > 31) {
    return 0;
  }else{
    return (operand1 >> operand2);
  }
}

//Function sra performs shift right arithmetic operation
unsigned long sra(unsigned long operand1, unsigned int amount) {
  unsigned long sign = (operand1 >> 31);
  unsigned long temp = 0;
  if ( amount == 0) {
    temp = operand1;
  }else if(amount > 31){
     if (sign == 0) {
      temp = 0;
    } else {
      temp = 0xffffffff;
    }
 
  } else {
    for ( unsigned int i = 31; i > 31- amount; i-- ) {
      temp = temp + power(2,i);
    }
    if (sign == 0) {
      temp = operand1 >> amount;
    } else {
      temp = temp |( operand1>>amount);
    }
  }
  return temp;
}


