// Instruction accurate C++ Model of COFFEE RISC processor
// Made by Marko Savolainen, TUT
// 18.2.2004
// Tampere University of Technology
// Institute of Digital and Computer Systems
// PO BOX 553
// FIN-33101 Tampere
// Finland
 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// --------------------------------------------------------------------------
// Revision history: (by Tuukka Kasanko)
// day.month.year
//  7.10.2004  Setting the PSR in the constructor and reset corrected.
//  8.10.2004  Moving the CCB fixed.
// 11.10.2004  interrupt(std::string int_name) implemented.
// 20.10.2004  'using namespace std' removed from the header file.
//  6.12.2004  Interrupts implemented into the step function.
//   4.1.2005  SWM instruction corrected,
//             16-bit mode in step function-corrected
//  11.1.2005  Reset value for INT_MASK in the CCB corrected.
//  24.1.2005  Jmp-instruction target address corrected.
//  25.1.2005  get_pc() interface fuction added.
//  26.1.2005  Reset values of the PSR_REGS changed due to changes in the 
//             COFFEE core.
//  28.1.2005  Functionality for timers implemented.
//  14.6.2005  PSR update bug in the RETU instruction corrected.
// --------------------------------------------------------------------------

#include "coffee_model.h"
#include "memory_models.h"
#include "arith_oper.h"
#include "topwindow.h"
#include <iostream>
#include <cstdlib>
#include <cmath>
#include <string>
#include <fstream>
#include <iomanip>

using namespace std;

//function to convert 32 bit binary string to decimal
unsigned long coffee_model::convert_32bit_string_to_decimal(string value) {
  unsigned long temp = 0;
  for(int i=0; i<32 ;i++) {

    if(value[i] == '1') { 
      temp += power(2,31-i);
    }
  } 
  return temp;
}

//function to convert 32 bit decima| to string
string  coffee_model::convert_32bit_decimal_to_string(unsigned long value){ 
  string temp = "";
  for(int i=0; i<32 ;i++) {

    if((value & power(2,31-i))  != 0) { 
      temp = temp + '1';
    }else{
      temp = temp + '0';
    }
  } 
  return temp;
}

//function to update conditional flags
void  coffee_model::update_flags(int C ,int Z ,int N, int reg_number){
  int temp = 0;
  if (C == 1 ) {
    flag_reg[reg_number] = temp | 0x0001;
  } else {
    flag_reg[reg_number] = temp & 0xfffe;
  }
  temp =flag_reg[reg_number];
  if (N == 1 ) {
    flag_reg[reg_number] = temp | 0x0002;
  } else {
    flag_reg[reg_number] = temp & 0xfffd;
  }
  temp =flag_reg[reg_number];
  if (Z == 1 ) {
    flag_reg[reg_number] = temp | 0x0004;
  } else {
    flag_reg[reg_number] = temp & 0xfffb;
  }
}



//function to update PSR variables
//See register specs for PSR bit locations
void  coffee_model::update_PSR_variables( unsigned long new_PSR) {
  unsigned long temp = new_PSR;
  if ((temp & 0x00000001)> 0) {
    user_superuser = true;
  } else {
    user_superuser = false;
  }
  if (((temp & 0x00000002)) > 0) {
    regtoread = 1; 
  } else {
    regtoread = 0;
  }
  if (((temp & 0x00000004)) > 0) {
    regtowrite = 1;
  } else {
    regtowrite = 0;
  }
  if (((temp & 0x00000008)) > 0) {
    mode = 32;
  } else {
    mode = 16;
  }
  if (((temp & 0x00000010)) > 0) {
    interrupts_enabled = true;
  } else {
    interrupts_enabled = false;
  }
  PSR = new_PSR;
}


//function which writes to register(tables suser_reg and user_reg)  
bool  coffee_model::write_to_register(int number, unsigned long value, int reg_type) {
  if(number != -1) {
    if(reg_type == 1) {
      if(number != 29) {// cant overwrite PSR
        suser_reg[number] = value;
      }
    } else {
      user_reg[number] = value;
    }
  }
  return true;
}

//function which reads from register(tables suser_reg and user_reg)
unsigned long  coffee_model::read_from_register(int number, int reg_type) {
  if(reg_type == 1) {
    return suser_reg[number];    
  } else {
    return user_reg[number];
  }
}

//function to check the correctness of i-address

bool coffee_model::check_instruction_address() {
  
 // checking instruction address violation 
 // See register specs for more information about allowed memory areas.
  if(user_superuser) {
    if(((PCB_REGS[0x1a] <=  PC && PCB_REGS[0x1b] >= PC) && ((PCB_REGS[0x1c] && 0x00000001) != 0)) ||
	((PCB_REGS[0x1a] >=  PC && PCB_REGS[0x1b] <= PC) && ((PCB_REGS[0x1c] && 0x00000001) == 0)))  {
      exception_occured = true;
      ecause = I_ADDR_VIOL;
      return false;
    }
  }
  
  //checking miss aligned jump(if the jump address is not aligned to word boundary(32-bit mode))
  if(mode == 32) {
    if ( (PC & 0x00000003) > 0) {
       exception_occured = true;          
       ecause = MISS_ALIGN_JUMP;
       return false;
    }
    
  }else{//(if the jump address is not aligned to halfword boundary(16-bit mode)
    if ( PC & 0x00000001 > 0) {
      exception_occured = true;
      return false;
    }
  }
  return true;  

} 
 

//function to convert register number from string to integer format 
int  coffee_model::reg_number(string to_convert) {
  int value = 0;
  for ( int i = 4; i>=0;i--) {
    if (to_convert[i] == '1') {
      value = value + power(2,4-i);
    } else {
      value = value;
    }
  }
  return value;
}

// function that writes data from memory
bool coffee_model::write_to_memory(unsigned long address, unsigned long writedata, memory_models & memmi) {

  // See register specs for more information about PCB values and registers
  // Checking if the writing points to PCB regs
  if ( address >= PCB_REGS[0] && address <= PCB_REGS[0] + 255) {
    unsigned int temp = address - PCB_REGS[0];
     switch(temp)  {
       case 0 :
	 PCB_REGS[0] = writedata;
	 break;
       case 1 :
	 PCB_REGS[1] = writedata;
	 break;
       case 2 :
	 PCB_REGS[2] = writedata;
	 break;
       case 3 :
	 PCB_REGS[3] = writedata;
	 break;
       case 4 :
	 PCB_REGS[4] = writedata;
	 break;
       case 5:
	 PCB_REGS[5] = writedata;
	 break;
       case 6 :
	 PCB_REGS[6] = writedata;
	 break;
       case 7 :
	 PCB_REGS[7] = writedata;
	 break;
       case 8 :
	 PCB_REGS[8] = writedata;
	 break;
       case 9 :
	 PCB_REGS[9] = writedata;
	 break;
       case 10 :
	 PCB_REGS[10] = writedata;
	 break;
       case 11:
	 PCB_REGS[11] = writedata;
	 break;
       case 12 :
	 PCB_REGS[12] = writedata;
	 break;
       case 13 :
	 PCB_REGS[13] = writedata;
	 break;
       case 14:
	 PCB_REGS[14] = 0x00000fff & writedata;
	 break;
       case 15 :
	 PCB_REGS[15] = 0x00000fff & writedata;
	 break;
       case 16 :
	 PCB_REGS[16] = 0x00000fff & writedata;
	 break;
       case 17:
	 PCB_REGS[17] = 0x00000fff & writedata;
	 break;
       case 18 :
	 PCB_REGS[18] = 0x00000fff & writedata;
	 break;
       case 19 :
	 PCB_REGS[19] = writedata;
	 break;
       case 20 :
	 PCB_REGS[20] = 0x0000ffff & writedata;
	 break;
       case 21 ://read only register	 
	 break;
       case 22 ://read only register	 
	 break;
       case 23 ://read only register	 
	 break;
       case 24:
	 PCB_REGS[24] = writedata;
	 break;
       case 25 :
	 PCB_REGS[25] = writedata;
	 break;
       case 26 :
	 PCB_REGS[26] = writedata;
	 break;
       case 27:
	 PCB_REGS[27] = writedata;
	 break;
       case 28 :
	 PCB_REGS[28] = 0x3 & writedata;
	 break;
       case 29 :
	 PCB_REGS[29] = writedata;
	 break;
       case 30 :
	 PCB_REGS[30] = writedata;
	 break;
       case 31:
	 PCB_REGS[31] = 0x000fff & writedata;
	 break;
       case 32:
	 PCB_REGS[32] = 0x0fffffff & writedata;
	 break;
       case 33:
	 PCB_REGS[33] = writedata;
	 break;
       case 34:
	 PCB_REGS[34] = writedata;
	 break;
       case 35:
	 PCB_REGS[35] = writedata;
	 break;
       case 36:
	 PCB_REGS[36] = writedata;
	 break;
       case 37:
	 PCB_REGS[37] = writedata;
	 break;
       case 38:
	 PCB_REGS[38] = writedata;
	 break;
       case 39:
	 PCB_REGS[39] = 0xff & writedata;
	 break;
       case 40:
	 PCB_REGS[40] = 0x7 & writedata;
	 break;
       default:
         break; 
     }
     return true;
     
  }else{//if not writing to memory
       
    return memmi.write_data_mem(address,convert_32bit_decimal_to_string(writedata)); 
  }
  
}


// function that reads data from memory
bool coffee_model::read_from_memory(int memory_read_reg, unsigned long address, memory_models & memmi) {
  // See register specs for more information about PCB values and registers
  //Checking if the reading points to PCB regs
  if ( address >= PCB_REGS[0] && address <= PCB_REGS[0] + 255) {
    unsigned int temp = address - PCB_REGS[0];
     switch(temp)  {
       case 0 :	 
	 write_to_register(memory_read_reg, PCB_REGS[0],regtowrite);
	 break;
       case 1 :
	 write_to_register(memory_read_reg, PCB_REGS[1],regtowrite);
	 break;
       case 2 :
	 write_to_register(memory_read_reg, PCB_REGS[2],regtowrite);
	 break;
       case 3 :
	 write_to_register(memory_read_reg, PCB_REGS[3],regtowrite);
	 break;
       case 4 :
	 write_to_register(memory_read_reg, PCB_REGS[4],regtowrite);
	 break;
       case 5:
	 write_to_register(memory_read_reg, PCB_REGS[5],regtowrite);
	 break;
       case 6 :
	 write_to_register(memory_read_reg, PCB_REGS[6],regtowrite);
	 break;
       case 7 :
	 write_to_register(memory_read_reg, PCB_REGS[7],regtowrite);
	 break;
       case 8 :
	 write_to_register(memory_read_reg, PCB_REGS[8],regtowrite);
	 break;
       case 9 :
	 write_to_register(memory_read_reg, PCB_REGS[9],regtowrite);
	 break;
       case 10 :
	 write_to_register(memory_read_reg, PCB_REGS[10],regtowrite);
	 break;
       case 11:
	 write_to_register(memory_read_reg, PCB_REGS[11],regtowrite);
	 break;
       case 12 :
	 write_to_register(memory_read_reg, PCB_REGS[12],regtowrite);
	 break;
       case 13 :
	 write_to_register(memory_read_reg, PCB_REGS[13],regtowrite);
	 break;
       case 14:
	 write_to_register(memory_read_reg, PCB_REGS[14],regtowrite);
	 break;
       case 15 :
	 write_to_register(memory_read_reg, PCB_REGS[15],regtowrite);
	 break;
       case 16 :
	 write_to_register(memory_read_reg, PCB_REGS[16],regtowrite);
	 break;
       case 17:
	 write_to_register(memory_read_reg, PCB_REGS[17],regtowrite);
	 break;
       case 18 :
	 write_to_register(memory_read_reg, PCB_REGS[18],regtowrite);
	 break;
       case 19 :
	 write_to_register(memory_read_reg, PCB_REGS[19],regtowrite);
	 break;
       case 20 :
	 write_to_register(memory_read_reg, PCB_REGS[20],regtowrite);
	 break;
       case 21:
	 write_to_register(memory_read_reg, PCB_REGS[21],regtowrite);
	 break;
       case 22 :
	 write_to_register(memory_read_reg, PCB_REGS[22],regtowrite);
	 break;
       case 23 :
	 write_to_register(memory_read_reg, PCB_REGS[23],regtowrite);
	 break;
       case 24:
	 write_to_register(memory_read_reg, PCB_REGS[24],regtowrite);
	 break;
       case 25 :
	 write_to_register(memory_read_reg, PCB_REGS[25],regtowrite);
	 break;
       case 26 :
	 write_to_register(memory_read_reg, PCB_REGS[26],regtowrite);
	 break;
       case 27 :
	 write_to_register(memory_read_reg, PCB_REGS[27],regtowrite);
	 break;
       case 28 :
	 write_to_register(memory_read_reg, PCB_REGS[28],regtowrite);
	 break;
       case 29:
	 write_to_register(memory_read_reg, PCB_REGS[29],regtowrite);
	 break;
       case 30:
	 write_to_register(memory_read_reg, PCB_REGS[30],regtowrite);
	 break;
       case 31:
	 write_to_register(memory_read_reg, PCB_REGS[31],regtowrite);
	 break;
       case 32:
	 write_to_register(memory_read_reg, PCB_REGS[32],regtowrite);
	 break;
       case 33:
	 write_to_register(memory_read_reg, PCB_REGS[33],regtowrite);
	 break;
       case 34:
	 write_to_register(memory_read_reg, PCB_REGS[34],regtowrite);
	 break;
       case 35:
	 write_to_register(memory_read_reg, PCB_REGS[35],regtowrite);
	 break;
       case 36:
	 write_to_register(memory_read_reg, PCB_REGS[36],regtowrite);
	 break;	
       case 37:
	 write_to_register(memory_read_reg, PCB_REGS[37],regtowrite);
	 break;
       case 38:
	 write_to_register(memory_read_reg, PCB_REGS[38],regtowrite);
	 break;
       case 39:
	 write_to_register(memory_read_reg, PCB_REGS[39],regtowrite);
	 break;
       case 40:
	 write_to_register(memory_read_reg, PCB_REGS[40],regtowrite);
	 break;
       default:
	 write_to_register(memory_read_reg, 0,regtowrite);
	 break;
     }
     return true;
  }else{
    //reading value from the memory
    string temp2 = memmi.read_data_mem(address);
    if(temp2 != "ERROR") { 
      unsigned long truevalue = convert_32bit_string_to_decimal(temp2); 	
      write_to_register(memory_read_reg, truevalue, regtowrite);
      return true;
    } else {
      cerr << "Error at PC: " << PC << ". Data memory does not contain anything at address: " << address
	   << ". PSR: " << PSR << std::endl;
      return false;
    }
  }
  
}

//function to convert immediate from string to long format 
long coffee_model::convert_signed_immediate(string to_convert) {
  long temp = 0;
  for ( int i = to_convert.length()-1; i>=0;i--) {
    if (to_convert[i] == '1') {
      temp = temp + power(2,to_convert.length()-1-i);
    } else {
      temp = temp;
    }
  }
  // if positive
  if ( to_convert[0] == '0') {
    return temp;
  } else { //if negative
    long sign = 0xffffffff;
    sign = sign << to_convert.length();
    temp = temp | sign;
    return temp;
  }
}

//function to convert immediate from string to unsigned long format 
unsigned long coffee_model::convert_unsigned_immediate(string to_convert) {
  long temp = 0;
  for ( int i = to_convert.length()-1; i>=0;i--) {
    if (to_convert[i] == '1') {
      temp = temp + power(2,to_convert.length()-1-i);
    } else {
      temp = temp;
    }
  }
  return temp;
}


// Checking if conditional execution is done
bool coffee_model::conditional_execution(int creg, int condition) {
  // See register specs for more information about condional registers
  //and bit locations inside it
  int flags = flag_reg[creg];
  if ( condition == 0) {//C=1
    if((flags & 0x0001) > 0) {
      return true;
    } else {
      return false;
    }
  }

  if ( condition == 1) {//Z=1 or N=0
    if(((flags & 0x0004) > 0) or ((flags & 0x0002) == 0)) {
      return true;
    } else {
      return false;
    }
  }
  
  if ( condition == 2) {//Z=1 or N=1
    if(((flags & 0x0004) > 0) or ((flags & 0x0002) > 0)) {
      return true;
    } else {
      return false;
    }
  }

  if ( condition == 3) {//Z=1
    if((flags & 0x0004) > 0) {       
      return true;
    } else {
      return false;
    }
  }

  if ( condition == 4) {//Z=0 and N=0
    if(((flags & 0x0004) == 0) and ((flags & 0x0002) == 0)) {
      return true;
    } else {
      return false;
    }
  }
  
  if ( condition == 5) {// N=1
    if(((flags & 0x0002) > 0)) {
      return true;
    } else {
      return false;
    }
  }

  if ( condition == 6) {//Z=0
    if((flags & 0x0004) == 0) {
      return true;
    } else {
      return false;
    }
  }
  if ( condition == 7) {//C=0
    if((flags & 0x0001) == 0) {
      return true;
    } else {
      return false;
    }
  }
  return true;
}



//Checking if the instruction is going to be executed
bool coffee_model::check_if_execute(INSTRUCTIONS inst, string instruction) {
  if( inst == CHRS or inst == CMP or
      inst == CMPI or inst == COP or inst == DI or inst == EI or inst == JAL or 
      inst == JMP or inst == LLI or inst == LUI or inst == NOP or 
      inst == RCON or inst == RETI or inst == RETU or inst == SCON or 
      inst == SWM or inst == TRAP) {
    return true; //no conditional execution-> always executed
  }else if((instruction[6] == '1' && mode == 32) && not( inst == BC or inst == BEGT or inst == BELT or inst == BEQ 
         or inst == BGT or inst == BLT or inst == BNE or inst == BNC)) {
    int creg = convert_unsigned_immediate(instruction.substr(7,3));
    int condition = convert_unsigned_immediate(instruction.substr(10,3));
    //Checking if the instruction should be executed
    return conditional_execution(creg, condition);
  }else if(inst == BC or inst == BEGT or inst == BELT or inst == BEQ 
	   or inst == BGT or inst == BLT or inst == BNE or inst == BNC){
    //Branches need always checking if they shold be executed
    int creg = 0;
    int condition = convert_unsigned_immediate(instruction.substr(3,3));
    if(mode == 32) {
      creg = convert_unsigned_immediate(instruction.substr(7,3));
    }
    //Checking if the instruction should be executed
    return conditional_execution(creg, condition);
    
      
  } else {
    //no condional execution-> instruction will be executed
    return true;
  }

}

//function that executes add instruction 
bool coffee_model::execute_add(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  bool dummy = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = add(long(read_from_register(sreg1,regtoread)),
	     long(read_from_register(sreg2,regtoread)),oflow,uflow,C,Z,N,dummy);
  
  //checking if overlow occured
  if ( uflow or oflow ) {
    exception_occured = true;
    ecause = OFLOW;
    return false;
  } else {
    update_flags(C,Z,N, 0);
    return write_to_register(dreg, unsigned(temp),regtowrite);
  }
    
}

//function that executes addi instruction 
bool coffee_model::execute_addi(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  bool uflow = false;
  bool oflow = false;
  bool dummy = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
    
  }
  
  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_signed_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_signed_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_signed_immediate(inst.substr(6,7));
  }
  
  temp = add(long(read_from_register(sreg1,regtoread)),immediate,oflow,uflow,C,Z,N,dummy );
  //checking if overlow occured
  if ( uflow or oflow ) {
    exception_occured = true;
    ecause = OFLOW;
    return false;
  } else {
    update_flags(C,Z,N, 0);
    return write_to_register(dreg, unsigned(temp),regtowrite);
  }
    
}

//function that executes addiu instruction 
bool coffee_model::execute_addiu(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  bool uflow = false;
  bool oflow = false;
   bool dummy = false;
  unsigned long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
    
  }
  
  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_unsigned_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_unsigned_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_unsigned_immediate(inst.substr(6,7));
  }
  temp = add(long(read_from_register(sreg1,regtoread)), long(immediate),oflow,uflow,C,Z,N,dummy);
  
  update_flags(C,Z,N, 0);
  return write_to_register(dreg, unsigned(temp),regtowrite);
     
}

//function that executes addu instruction 
bool coffee_model::execute_addu(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  bool dummy = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = add(long(read_from_register(sreg1,regtoread)),
	     long(read_from_register(sreg2,regtoread)),oflow,uflow,C,Z,N,dummy);
  update_flags(C,Z,N, 0);
  return write_to_register(dreg, unsigned(temp),regtowrite);
}


//function that executes and instruction 
bool coffee_model::execute_and(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = and_f(read_from_register(sreg1,regtoread),
	       read_from_register(sreg2,regtoread));
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes andi instruction 
bool coffee_model::execute_andi(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long temp = 0;
  unsigned long immediate = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
  }


  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_unsigned_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_unsigned_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_unsigned_immediate(inst.substr(6,7));
  }
  temp = and_f(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
}



//function that executes or instruction 
bool coffee_model::execute_or(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }

  temp = or_f(read_from_register(sreg1,regtoread),
	      read_from_register(sreg2,regtoread));
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes ori instruction 
bool coffee_model::execute_ori(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long temp = 0;
  unsigned long immediate = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
  }


  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_unsigned_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_unsigned_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_unsigned_immediate(inst.substr(6,7));
  }
  temp = or_f(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes xor instruction 
bool coffee_model::execute_xor(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = xor_f(read_from_register(sreg1,regtoread),
	       read_from_register(sreg2,regtoread));
  return write_to_register(dreg, temp,regtowrite);
}


//function that executes bc instruction 
bool coffee_model::execute_bc(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }
  
  
    
   //decoding the immediate value
   if (mode == 32) { 
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;

    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }
   new_PC = temp;
   jump_occured[1] = true;
   if (j_oflow ) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
 
   return true;
}

//function that executes begt instruction 
bool coffee_model::execute_begt(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }
  
 
   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }


   new_PC = temp;
   jump_occured[1] = true;
  
   if (j_oflow ) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
   
    
   return true;
}


//function that executes belt instruction 
bool coffee_model::execute_belt(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }
  
  
    
   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }
  
   
   new_PC = temp;
   jump_occured[1] = true;
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
   
   return true;
 
}


//function that executes beq instruction 
bool coffee_model::execute_beq(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }

   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }
 
   
   new_PC = temp;
   jump_occured[1] = true;
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
   
   return true;
 
}

//function that executes bgt instruction 
bool coffee_model::execute_bgt(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }
  
   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }

   new_PC = temp;
   jump_occured[1] = true;
  
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
   
   return true;
  
}


//function that executes begt instruction 
bool coffee_model::execute_blt(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
   bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  }

   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }

   new_PC = temp;
   jump_occured[1] = true;
    
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
    
   return true;
}

//function that executes bne instruction 
bool coffee_model::execute_bne(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  } 
  
   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }

   new_PC = temp;
   jump_occured[1] = true;
  
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
  
   return true;
}

//function that executes bnc instruction 
bool coffee_model::execute_bnc(string inst) {
  int creg  = 0;
  unsigned long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
  } else { //16-bit mode
    creg = 0;
  } 
  
   //decoding the immediate value
   if (mode == 32) {
     immediate = convert_signed_immediate(inst.substr(10,22));
   } else {
     immediate =  convert_signed_immediate(inst.substr(6,10));
   }
   //shifting one to left(see specs)
   immediate = immediate << 1;
    if (mode == 32){
      temp = add(PC+4,immediate,oflow,uflow,Z,C,N,j_oflow);
    }else{
      temp = add(PC+2,immediate,oflow,uflow,Z,C,N,j_oflow);
    }

   new_PC = temp;
   jump_occured[1] = true;
  
   if (j_oflow) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
   }
  
   return true;
}



//function that executes chrs instruction 
bool coffee_model::execute_chrs(string inst) {
 
  if(user_superuser)  {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }


  int value  = 0;
  if (mode == 32) {
    value  = reg_number("000" + inst.substr(20,2));
  } else { //16-bit mode
    value  = reg_number("000" + inst.substr(11,2));
  }
  
  switch (value) {
    case 0:
      regtoread = 0;
      regtowrite = 0;	
      PSR = PSR & 0xfffffff9;
      suser_reg[29] = PSR;
      break;
    case 1:
      PSR = PSR & 0xfffffffb;
      PSR = PSR | 0x00000002;  
      suser_reg[29] = PSR;
      regtoread = 1;
      regtowrite = 0;	
      break;
    case 2:
      PSR = PSR & 0xfffffffd;
      PSR = PSR | 0x00000004;
      suser_reg[29] = PSR;
      regtoread = 0;
      regtowrite = 1;	
      break;
    default:
      PSR = PSR | 0x00000006;
      suser_reg[29] = PSR;
      regtoread = 1;
      regtowrite = 1;	
      break;
  } 
    
    return true;
}

//function that executes cmp instruction 
bool coffee_model::execute_cmp(string inst) {
  int creg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    creg  = 0;
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = reg_number("11" + inst.substr(13,3));
  }
  
  temp = sub(long(read_from_register(sreg1,regtoread)),
	     long(read_from_register(sreg2,regtoread)),oflow,uflow,C,Z,N, true);
  update_flags(C,Z,N, creg);
  return true;
    
}

//function that executes cmpi instruction 
bool coffee_model::execute_cmpi(string inst) {
  int creg  = 0;
  int sreg1 = 0;
  bool uflow = false;
  bool oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    creg  = reg_number("00" + inst.substr(7,3));
    sreg1 = reg_number(inst.substr(22,5));
    temp = convert_signed_immediate((inst.substr(27,5) + inst.substr(10,12)));
  } else { //16-bit mode
    creg  = 0;
    sreg1 = reg_number("11" + inst.substr(6,3));
    temp = convert_signed_immediate(inst.substr(9,7));
  }
  
  temp = sub(long(read_from_register(sreg1,regtoread)),temp,oflow,uflow,C,Z,N,true);
  update_flags(C,Z,N, creg);
  return true;
    
}

//function that executes conb instruction 
bool coffee_model::execute_conb(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  
  temp = conb(read_from_register(sreg1,regtoread),
	      read_from_register(sreg2,regtoread));
  
  return write_to_register(dreg, temp,regtowrite);
  
    
}

//function that executes conh instruction 
bool coffee_model::execute_conh(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg2 = reg_number("11" + inst.substr(6,3));
    sreg1 = dreg;
  }
  // changes in register order due specs modifications
  temp = conh(read_from_register(sreg2,regtoread),
	      read_from_register(sreg1,regtoread));
  
  return write_to_register(dreg, temp,regtowrite);
  
    
}


//function that executes conh instruction 
bool coffee_model::execute_cop(string inst) {
  unsigned long value = 0;
  int cop_number = 0;
  if(mode != 32)  {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }
  value = convert_unsigned_immediate(inst.substr(8,24));
  cop_number = reg_number("000" + inst.substr(6,2)); 
   //see PCB specs for register index
   if(cop_number == 0) {
     coprocessor0regs[(0x0000001f & PCB_REGS[0x20])] = value;
   }else if(cop_number == 1) {
     coprocessor1regs[((0x000003e0 & PCB_REGS[0x20])>>5)] = value;
   }else if(cop_number == 2) {
     coprocessor2regs[((0x00007c00 & PCB_REGS[0x20])>>10)] = value;
   }else{
     coprocessor3regs[((0x000f8000 & PCB_REGS[0x20])>>15)] = value;
   }
  return true; 
    
}


//function that executes di instruction 
bool coffee_model::execute_di() {
  if(user_superuser)  {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }
  interrupts_enabled = false;
  // setting the register value also
  PSR = PSR  & 0xffffffef;
  suser_reg[29] = suser_reg[29] & 0xffffffef;
  return true; 
    
}

//function that executes ei instruction 
bool coffee_model::execute_ei() {
  if(user_superuser)  {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }
  interrupts_enabled = true;
  // setting the register value also
  PSR = PSR  | 0x00000010;
  suser_reg[29] = suser_reg[29] | 0x00000010;
  return true; 
    
}

//function that executes exb instruction 
bool coffee_model::execute_exb(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    
  }
  
  //decoding the immediate value
  if (mode == 32) {  
    immediate = convert_unsigned_immediate(inst.substr(20,2));
  } else {
    immediate =  convert_unsigned_immediate(inst.substr(11,2));
  }
  
  temp = exb(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
     
}


//function that executes exbf instruction 
bool coffee_model::execute_exbf(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 =0;
  unsigned long immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  //decoding the immediate value
  if (mode == 32) {  
    immediate = convert_unsigned_immediate(inst.substr(17,2));
  } else {
    immediate =  convert_unsigned_immediate(inst.substr(11,2));
  }
  
  temp = exbf(read_from_register(sreg1,regtoread),
	      read_from_register(sreg2,regtoread));
  return write_to_register(dreg, temp,regtowrite);
     
}

//function that executes exbfi instruction 
bool coffee_model::execute_exbfi(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;    
  } 
  //decoding the immediate value 
  immediate = convert_unsigned_immediate(inst.substr(11,11));
  temp = exbf(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
     
}


//function that executes exh instruction 
bool coffee_model::execute_exh(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
     exception_occured = true;
     ecause = ILLEG_OPCODE;
     return false; 
  }
  
  //decoding the immediate value 
  if ( mode == 32) {
    if ( inst[21] == '1') {
      immediate = 1;
    } else {
      immediate = 0;
    }
  } else {
    if ( inst[12] == '1') {
      immediate = 1;
    } else {
      immediate = 0;
    }
  }

  temp = exh(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
     
}

//function that executes jal instruction 
bool coffee_model::execute_jal(string inst) {
  signed long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool  j_oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    temp = convert_signed_immediate(inst.substr(7,25));
  } else { //16-bit mode
    temp = convert_signed_immediate(inst.substr(6,10));
  }
  
  temp = temp << 1;
  if (mode == 32){
    temp = add(PC+4,temp,oflow,uflow,Z,C,N,j_oflow);
  }else{
    temp = add(PC+2,temp,oflow,uflow,Z,C,N,j_oflow);
  }

  //Saving the link address
  if ( mode == 32) {
     write_to_register(31,PC + 8,regtowrite);
  } else {
      write_to_register(31,PC + 4,regtowrite);
  }
  new_PC = temp;
  jump_occured[1] = true;
  if (j_oflow and not(exception_occured)) {
     jmp_oflow = true;
     exception_occured = true;
     ecause = J_ADDR_OFLOW;
     return false;
  }
  
  return true;
    
}


//function that executes jalr instruction 
bool coffee_model::execute_jalr(string inst) {
  unsigned long temp = 0;
  int sreg1 = 0;
  if (mode == 32) {
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  
  temp = read_from_register(sreg1,regtoread); 
  
  //Saving the link address
  if ( mode == 32) {
    write_to_register(31,PC + 8,regtowrite);
  } else {
    write_to_register(31,PC + 4,regtowrite);
  }
  new_PC = temp;
  jump_occured[1] = true;
  return true;
    
}

//function that executes jmp instruction 
bool coffee_model::execute_jmp(string inst) {
  signed long temp = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  if (mode == 32) {
    temp = convert_signed_immediate(inst.substr(7,25));
  } else { //16-bit mode
    temp = convert_signed_immediate(inst.substr(6,10));
  }
  
  temp = temp << 1;

  if (mode == 32) // 32-bit mode
  {  
    temp = add((PC + 4),temp,oflow,uflow,Z,C,N,j_oflow);
  }
  else // 16-bit mode
  {
    temp = add((PC + 2),temp,oflow,uflow,Z,C,N,j_oflow);
  }

  new_PC = temp;

  jump_occured[1] = true;
  
  if (j_oflow) {
    jmp_oflow = true;
    exception_occured = true;
    ecause = J_ADDR_OFLOW;
    return false;
  }
 
  return true;
    
}

//function that executes jmpr instruction 
bool coffee_model::execute_jmpr(string inst) {
  unsigned long temp = 0;
  int sreg1 = 0;
  if (mode == 32) {
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  
  temp = read_from_register(sreg1,regtoread); 
  
  new_PC = temp;
  jump_occured[1] = true;
  
  return true;
    
}


//function that executes ld instruction 
bool coffee_model::execute_ld(string inst, memory_models & memclass) {
  int dreg  = 0;
  int sreg1 = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow = false;
  signed long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  
  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_signed_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_signed_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_signed_immediate(inst.substr(9,4));
  }

   temp = add(long(read_from_register(sreg1,regtoread)), immediate,oflow,uflow,C,Z,N,j_oflow);
 
  //daddrviol check
  if( user_superuser) { //if usermode
    if( (unsigned(temp) >= PCB_REGS[0x18] && unsigned(temp) <= PCB_REGS[0x19]) && ((PCB_REGS[0x1c] && 0x00000002) !=  0)|| 
        (unsigned(temp) <= PCB_REGS[0x18] && unsigned(temp) >= PCB_REGS[0x19]) && ((PCB_REGS[0x1c] && 0x00000002) ==  0)) {
       exception_occured = true;
       ecause = D_ADDR_VIOL;
       return false;
    }
  }
 
  if (j_oflow) {
    exception_occured = true;
    ecause = D_ADDR_OFLOW;
    return false;
  }
      
  return read_from_memory(dreg,temp,memclass);
 

     
}


//function that executes lli instruction 
bool coffee_model::execute_lli(string inst) {
  int dreg  = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number( inst.substr(27,5));
    temp = convert_unsigned_immediate(inst[22] + inst.substr(7,15));
  } else {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }
  
 
  return write_to_register(dreg,(temp & 0x0000ffff),regtowrite);
    
}

//function that executes lui instruction 
bool coffee_model::execute_lui(string inst) { 
      
  int dreg  = 0;
  unsigned long temp = 0;
  unsigned long temp2 = 0;
  if (mode == 32) {
    dreg  = reg_number( inst.substr(27,5));
    temp = convert_unsigned_immediate(inst[22] + inst.substr(7,15));
  } else {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false;
  }
 
  //registers have to from the same register set-> using write set 
  temp2 = 0x0000ffff &  read_from_register(dreg,regtowrite);
  temp = temp  << 16;
  return write_to_register(dreg,(temp & 0xffff0000)+ temp2,regtowrite);
    
}


//function that executes mov instruction 
bool coffee_model::execute_mov(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  
  return write_to_register(dreg, read_from_register(sreg1,regtoread),regtowrite);
  
    
}



//function that executes movtc instruction 
bool coffee_model::execute_movtc(string inst) {
   int cop_dreg  = 0;
   int sreg1 = 0;
   int cop_number = 0;
   if (mode == 32) {
     cop_dreg  = reg_number(inst.substr(15,5));
     sreg1 = reg_number(inst.substr(22,5));
     cop_number = reg_number("000"+  inst.substr(20,2));
   } else { //16-bit mode
     cop_dreg  = reg_number(inst.substr(9,5));
     sreg1 = reg_number("11" + inst.substr(6,3));
     cop_number = reg_number("000"+  inst.substr(14,2));
   }
   if(cop_number == 0) {
      coprocessor0regs[cop_dreg] = read_from_register(sreg1,regtoread);
   }else if(cop_number == 1) {
      coprocessor1regs[cop_dreg] = read_from_register(sreg1,regtoread);
   }else if(cop_number == 2) {
      coprocessor2regs[cop_dreg] = read_from_register(sreg1,regtoread);
   }else{
      coprocessor3regs[cop_dreg] = read_from_register(sreg1,regtoread);
   } 

  return true;
}

//function that executes movcf instruction 
bool coffee_model::execute_movcf(string inst) {
   int cop_sreg  = 0;
   int dreg = 0;
   unsigned long value = 0;
   int cop_number = 0;

   if (mode == 32) {
     cop_sreg  = reg_number(inst.substr(15,5));
     dreg = reg_number(inst.substr(27,5));
     cop_number = reg_number("000"+  inst.substr(20,2));
   } else { //16-bit mode
     cop_sreg  = reg_number( inst.substr(6,5));
     dreg = reg_number("11" + inst.substr(13,3));
     cop_number = reg_number("000"+  inst.substr(11,2));
   }
   if(cop_number == 0) {
      value = coprocessor0regs[cop_sreg];
   }else if(cop_number == 1) {
      value = coprocessor1regs[cop_sreg];
   }else if(cop_number == 2) {
      value = coprocessor2regs[cop_sreg];
   }else{
      value = coprocessor3regs[cop_sreg];
   }
   return write_to_register(dreg,value ,regtowrite);

}


//function that executes mulhi instruction 
bool coffee_model::execute_mulhi(string inst) {
  int dreg  = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
  }
  
  return write_to_register(dreg, mulhi_result,regtowrite);      
}


//function that executes muli instruction 
bool coffee_model::execute_muli(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  signed long immediate = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  signed long sign_result = 0;
  signed long sign_hi = 0;
  int sign = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
    
  }
  
  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_signed_immediate(inst.substr(13,9));
    } else {// cex = 0 -> longer immediate
      immediate = convert_signed_immediate(inst.substr(7,15));
    }
  } else {
      immediate =  convert_signed_immediate(inst.substr(6,7));
  }

  temp = multiply_32bit(convert_sign_unsign(read_from_register(sreg1,regtoread)),
			convert_sign_unsign(immediate),mulhi_res);
  
  sign = (read_from_register(sreg1,regtoread) & 0x80000000) ^ (immediate & 0x80000000);
  
  //Checking if the result is negative
  if ( sign != 0 ) {
    sign_result =  mult_signed(sign, temp, mulhi_res , sign_hi);
    mulhi_result = unsigned(sign_hi);
    return write_to_register(dreg, unsigned(sign_result),regtowrite);
  } else {
    mulhi_result = mulhi_res;
    return write_to_register(dreg, temp,regtowrite);
  }
    
}



//function that executes muls instruction 
bool coffee_model::execute_muls(string inst) {
  
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  signed long sign_result = 0;
  signed long sign_hi = 0;
  int sign = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  
  temp = multiply_32bit(convert_sign_unsign(read_from_register(sreg1,regtoread)),
			convert_sign_unsign(read_from_register(sreg2,regtoread)),mulhi_res);
  
  sign = (read_from_register(sreg1,regtoread) & 0x80000000) ^ (read_from_register(sreg2,regtoread) & 0x80000000);
 
  //Checking if the result is negative
  if ( sign != 0 ) {
    sign_result =  mult_signed(sign, temp, mulhi_res , sign_hi);
    mulhi_result = unsigned(sign_hi);
    return write_to_register(dreg, unsigned(sign_result),regtowrite);
  } else {
    mulhi_result = mulhi_res;
    return write_to_register(dreg, temp,regtowrite);
  }
    
}


//function that executes muls_16 instruction 
bool coffee_model::execute_muls_16(string inst) {


  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  signed long operand1 = 0;
  signed long operand2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  signed long sign_result = 0;
  signed long sign_hi = 0;
  int sign = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  operand1 = 0x0000ffff & (read_from_register(sreg1,regtoread));
  operand2 = 0x0000ffff & (read_from_register(sreg2,regtoread));


  //sign extension
  if ((operand1 >> 15) == 1) {
    operand1 = operand1 | 0xffff0000;
  }
  if ((operand2 >> 15) == 1) {
    operand2 = operand2 | 0xffff0000;
  }
  
  temp = multiply_32bit(convert_sign_unsign(operand1),convert_sign_unsign(operand2),mulhi_res);
  
  sign = (operand1 & 0x80000000) ^ (operand2 & 0x80000000);
  //Checking if the result is negative
  if ( sign != 0 ) {
    sign_result =  mult_signed(sign, temp, mulhi_res , sign_hi);
    return write_to_register(dreg, unsigned(sign_result),regtowrite);
  } else {
    return write_to_register(dreg, temp,regtowrite);
  }
    
}



//function that executes mulu instruction 
bool coffee_model::execute_mulu(string inst) {

  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  
  temp = multiply_32bit(read_from_register(sreg1,regtoread),
			read_from_register(sreg2,regtoread),mulhi_res);
  
  mulhi_result = mulhi_res;
  return write_to_register(dreg, temp,regtowrite);
    
}


//function that executes mulu_16 instruction 
bool coffee_model::execute_mulu_16(string inst) {

  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  unsigned long operand1 = 0;
  unsigned long operand2 = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  operand1 = 0x0000ffff & (read_from_register(sreg1,regtoread));
  operand2 = 0x0000ffff & (read_from_register(sreg2,regtoread));
  
  temp = multiply_32bit(operand1,operand2,mulhi_res);
  return write_to_register(dreg, temp,regtowrite);    
}


//function that executes mulus instruction 
bool coffee_model::execute_mulus(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  signed long sign_result = 0;
  signed long sign_hi = 0;
  int sign = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  

  temp = multiply_32bit(read_from_register(sreg1,regtoread),
			convert_sign_unsign(read_from_register(sreg2,regtoread)),mulhi_res);
  
  sign = (read_from_register(sreg2,regtoread) & 0x80000000);
  
  //Checking if the result is negative
  if ( sign != 0 ) {
    sign_result =  mult_signed(sign, temp, mulhi_res , sign_hi);
    mulhi_result = unsigned(sign_hi);
    return write_to_register(dreg, unsigned(sign_result),regtowrite);
  } else {
    mulhi_result = mulhi_res;
    return write_to_register(dreg, temp,regtowrite);
  }
    
}

//function that executes mulus_16 instruction 
bool coffee_model::execute_mulus_16(string inst) {

  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long operand1 = 0;
  signed long operand2 = 0;
  unsigned long temp = 0;
  unsigned long mulhi_res = 0;
  signed long sign_result = 0;
  signed long sign_hi = 0;
  int sign = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  operand1 = 0x0000ffff & (read_from_register(sreg1,regtoread));
  operand2 = 0x0000ffff & (read_from_register(sreg2,regtoread));


  //sign extension
  if ((operand2 >> 15) == 1) {
    operand2 = operand2 | 0xffff0000;
  }
  
  temp = multiply_32bit(operand1,convert_sign_unsign(operand2),mulhi_res);
  
  sign = (operand2 & 0x00008000);
  
  //Checking if the result is negative
  if ( sign != 0 ) {
    sign_result =  mult_signed(sign, temp, mulhi_res , sign_hi);
    return write_to_register(dreg, unsigned(sign_result),regtowrite);
  } else {
    return write_to_register(dreg, temp,regtowrite);
  }
    
}


//function that executes not instruction 
bool coffee_model::execute_not(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  temp = not_f(read_from_register(sreg1,regtoread));
  return write_to_register(dreg, temp,regtowrite);
}


//function that executes rcon instruction 
bool coffee_model::execute_rcon(string inst) {
  int sreg1 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    sreg1 = reg_number("11" + inst.substr(6,3));
  }
  temp = read_from_register(sreg1,regtoread);
  flag_reg[0] = 0x00000007 & temp; 
  flag_reg[1] = (0x00000038 & temp) >> 3;
  flag_reg[2] = (0x000001c0 & temp) >> 6;
  flag_reg[3] = (0x00000e00 & temp) >> 9;
  flag_reg[4] = (0x00007000 & temp) >> 12;
  flag_reg[5] = (0x00038000 & temp) >> 15;
  flag_reg[6] = (0x001c0000 & temp) >> 18;
  flag_reg[7] = (0x00e00000 & temp) >> 21;
  return true;
}


//function that executes reti instruction 
bool coffee_model::execute_reti() {

  // Restore PSR from the hardware stack.
  // Restore PC from the hardware stack.
  // Restore CR0 from the hardware stack
  // Restore jump status from the interrupt stack.

  // PSR = interrupt_psr_stack[0];
  PSR = PCB_REGS[0x27];
  update_PSR_variables(PSR); // Update variables related to PSR.
  suser_reg[29] = PSR;

  PC = PCB_REGS[0x26];
  flag_reg[0] = PCB_REGS[0x28];

  jump_occured[0] = interrupt_jump_occurred_stack[0][0];
  jump_occured[1] = interrupt_jump_occurred_stack[0][1];
  for(unsigned int i = 1; i < 12; i++){
    interrupt_psr_stack[i-1] = interrupt_psr_stack[i];
    interrupt_pc_stack[i-1] = interrupt_pc_stack[i];
    interrupt_cr0_stack[i-1] = interrupt_cr0_stack[i];
    interrupt_jump_occurred_stack[i-1][0] = interrupt_jump_occurred_stack[i][0];
    interrupt_jump_occurred_stack[i-1][1] = interrupt_jump_occurred_stack[i][1];
  }

  // Get the interrupt number which flag should be cleared from
  // the INT_SERV register.
  unsigned int int_numb = interrupt_number_stack[0];

  // Shift the stack down by one.
  for(unsigned int i = 1; i < 12; i++){
    interrupt_number_stack[i-1] = interrupt_number_stack[i];  
  }  
  
  // Clear the correct bit from the INT_SERV register
  switch (int_numb){
  case 0:    
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffffe;
    break;
  case 1:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffffd;
    break;
  case 2:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffffb;
    break;
  case 3:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffff7;
    break;
  case 4:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xffffffef;
    break;
  case 5:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xffffffdf;
    break;
  case 6:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xffffffbf;
    break;
  case 7:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xffffff7f;
    break;
  case 8:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffeff;
    break;
  case 9:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffdff;
    break;
  case 10:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffffbff;
    break;
  case 11:
    PCB_REGS[0x11] = PCB_REGS[0x11] & 0xfffff7ff;
    break;
  }

  return true;
}


//function that executes retu instruction 
bool coffee_model::execute_retu() {

  if(user_superuser) {
    exception_occured = true;
    ecause = ILLEG_OPCODE;
    return false; 
  }
  //reading new address from suser_reg[31]
  regtoread = 1;
  jump_occured[1] = true;
  new_PC = read_from_register(31,regtoread);
  regtowrite = 1;
  psr_update_delayed[0] = true;

  return true;
}


//function that executes scall instruction 
bool coffee_model::execute_scall() {
  new_PC = PCB_REGS[0x1d];
  jump_occured[1] = true;
  suser_reg[30] = PSR;
  regtowrite = 1;
  write_to_register(30, suser_reg[30],1);
  if ( mode == 32) {
    write_to_register(31, PC + 8,regtowrite);
  } else {
    write_to_register(31, PC + 4,regtowrite);
  }

  // Set PSR according to the documentation.
  PSR = 14;
  update_PSR_variables(PSR);
  // Make the change in PSR visible to the user.
  suser_reg[29] = PSR;

  return true;
}


//function that executes scon instruction 
bool coffee_model::execute_scon(string inst) {
  int dreg = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg = reg_number(inst.substr(27,5));
  } else { //16-bit mode
    dreg = reg_number("11" + inst.substr(13,3));
  }
  temp = flag_reg[0] & 0x00000007; 
  temp = temp | ((flag_reg[1] & 0x00000007) << 3);
  temp = temp | ((flag_reg[2] & 0x00000007) << 6);
  temp = temp | ((flag_reg[3] & 0x00000007) << 9);
  temp = temp | ((flag_reg[4] & 0x00000007) << 12);
  temp = temp | ((flag_reg[5] & 0x00000007) << 15);
  temp = temp | ((flag_reg[6] & 0x00000007) << 18);
  temp = temp | ((flag_reg[7] & 0x00000007) << 21);

  return write_to_register(dreg, temp,regtowrite);

}


//function that executes sext instruction 
bool coffee_model::execute_sext(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
    
  }
  
  //decoding the immediate valiue
  immediate =  read_from_register(sreg2,regtoread) & 0x0000001f;
  
  temp = sext(read_from_register(sreg1,regtoread),immediate);
  
  return write_to_register(dreg,temp,regtowrite);
     
}


//function that executes sexti instruction 
bool coffee_model::execute_sexti(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long immediate = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    immediate = convert_unsigned_immediate(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
    immediate = convert_unsigned_immediate(inst.substr(8,5));
    
  }
  
  temp = sext(read_from_register(sreg1,regtoread),immediate);
  
  return write_to_register(dreg,temp,regtowrite);
     
}



//function that executes sll instruction 
bool coffee_model::execute_sll(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  int Z = 0;
  int N = 0;
  int C = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = sll(read_from_register(sreg1,regtoread),
	     ((read_from_register(sreg2,(regtoread)) & 0x0000003f)),Z,N,C);
  update_flags(C,Z,N, 0);
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes slli instruction 
bool coffee_model::execute_slli(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int Z = 0;
  int N = 0;
  int C = 0;
  unsigned long temp = 0;
  unsigned long immediate = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
  }

  //decoding the immediate value
  if (mode == 32) {
    immediate = convert_unsigned_immediate(inst.substr(16,6));
  } else {
    immediate =  convert_unsigned_immediate(inst.substr(6,6));
  }
  temp = sll(read_from_register(sreg1,regtoread),immediate,Z,N,C);
  update_flags(C,Z,N, 0);
  return write_to_register(dreg, temp,regtowrite);
}


//function that executes sra instruction 
bool coffee_model::execute_sra(string inst) {
 
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = sra(read_from_register(sreg1,regtoread),
	     (read_from_register(sreg2,regtoread) & 0x0000003f));
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes srai instruction 
bool coffee_model::execute_srai(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long temp = 0;
  unsigned long immediate = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
  }

  //decoding the immediate value
  if (mode == 32) {
    immediate = convert_unsigned_immediate(inst.substr(16,6));
  } else {
    immediate =  convert_unsigned_immediate(inst.substr(6,6));
  }
  temp = sra(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes srl instruction 
bool coffee_model::execute_srl(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  unsigned long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = srl(read_from_register(sreg1,regtoread),
	     (read_from_register(sreg2,regtoread) & 0x0000003f));
  return write_to_register(dreg, temp,regtowrite);
}

//function that executes srli instruction 
bool coffee_model::execute_srli(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  unsigned long temp = 0;
  unsigned long immediate = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = dreg;
  }

  //decoding the immediate value
  if (mode == 32) {
    immediate = convert_unsigned_immediate(inst.substr(16,6));
  } else {
    immediate =  convert_unsigned_immediate(inst.substr(6,6));
  }
  temp = srl(read_from_register(sreg1,regtoread),immediate);
  return write_to_register(dreg, temp,regtowrite);
}


//function that executes st instruction 
bool coffee_model::execute_st(string inst, memory_models & memclass) {
  int sreg1  = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  bool j_oflow =false;
  signed long immediate = 0;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = reg_number("11" + inst.substr(13,3));
  }
  
  //decoding the immediate value
  if (mode == 32) {
    // cex = 1 -> shorter immediate
    if(inst[6] == '1') {
      immediate = convert_signed_immediate(inst.substr(27,5) + inst.substr(13,4));
    } else {// cex = 0 -> longer immediate
      immediate = convert_signed_immediate(inst.substr(27,5) + inst.substr(7,10));
    }
  } else {
      immediate =  convert_signed_immediate(inst.substr(9,4));
  }
  
   temp = add(long(read_from_register(sreg1,regtoread)), immediate,oflow,uflow,C,Z,N,j_oflow);
  
  //daddrviol check
  if( user_superuser) { //if usermode
       if( (unsigned(temp) >= PCB_REGS[0x18] && unsigned(temp) <= PCB_REGS[0x19]) && ((PCB_REGS[0x1c] && 0x00000002) !=  0)|| 
        (unsigned(temp) <= PCB_REGS[0x18] && unsigned(temp) >= PCB_REGS[0x19]) && ((PCB_REGS[0x1c] && 0x00000002) ==  0)) {
       exception_occured = true;
       ecause = D_ADDR_VIOL;
       return false;
    }
  } 
  if (j_oflow) {
    exception_occured = true;
    ecause = D_ADDR_OFLOW;
    return false;
  }     
  write_to_memory(temp,read_from_register(sreg2,regtoread),memclass);
  return true;
}

//function that executes sub instruction 
bool coffee_model::execute_sub(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  
  temp = sub(long(read_from_register(sreg1,regtoread)),
	     long(read_from_register(sreg2,regtoread)),oflow,uflow,C,Z,N, false);
  
  if ( uflow or oflow ) {
    exception_occured = true;
    ecause = OFLOW;
    return false;
  } else {
    update_flags(C,Z,N, 0);
    return write_to_register(dreg, unsigned(temp),regtowrite);
  }
    
}

//function that executes subu instruction 
bool coffee_model::execute_subu(string inst) {
  int dreg  = 0;
  int sreg1 = 0;
  int sreg2 = 0;
  bool uflow = false;
  bool oflow = false;
  int Z = 0;
  int C = 0;
  int N = 0;
  signed long temp = 0;
  if (mode == 32) {
    dreg  = reg_number(inst.substr(27,5));
    sreg1 = reg_number(inst.substr(22,5));
    sreg2 = reg_number(inst.substr(17,5));
  } else { //16-bit mode
    dreg  = reg_number("11" + inst.substr(13,3));
    sreg1 = reg_number("11" + inst.substr(6,3));
    sreg2 = dreg;
  }
  temp = sub(long(read_from_register(sreg1,regtoread)),
	     long(read_from_register(sreg2,regtoread)),oflow,uflow,C,Z,N, false);
  update_flags(C,Z,N, 0);
  return write_to_register(dreg, unsigned(temp),regtowrite);
}

//function that executes swm instruction 
bool coffee_model::execute_swm(string inst) {

  if ( mode == 32) {
    if( inst[16] == '0')  { //32->16-bit 
      new_mode = 16;
      mode_change_delay = 1;
      mode_change = 0;
    }// else staying in 32-bit mode
  } else {
    if( inst[7] == '1')  { //16->32-bit 
      new_mode = 32;
      mode_change_delay = 0;
      mode_change = 0;
    }// else staying in 16-bit mode
  }

  return true;

}


//function that executes trap instruction 
bool coffee_model::execute_trap(string inst) {
  if ( mode == 32) {
    exception_occured = true;
    ecause = ETRAP;
    trap_value = convert_unsigned_immediate(inst.substr(17,5));
  } else {
    exception_occured = true;
    ecause = ETRAP;
    trap_value = convert_unsigned_immediate(inst.substr(8,5));
  }
  return false;
}

//function to find instruction from the binary string
INSTRUCTIONS coffee_model::which_instruction(string opcode) {
  
  string temp = opcode.substr(0,6);
  int value = 0;
  //Calculating number value of opcode for switch
  if(temp[0] == '1') {
    value += 32;
  }
  if(temp[1] == '1') {
    value += 16;
  }
  if(temp[2] == '1') {
    value += 8;
  }
  if(temp[3] == '1') {
    value += 4;
  }
  if(temp[4] == '1') {
    value += 2;
  }
  if(temp[5] == '1') {
    value += 1;
  }
  switch (value) {
    case 0:
      return ADDU;
    case 1:
      return ADD;
    case 2:
      return AND;
    case 3:
      return CONB;
    case 4:
      return CONH;
    case 5:
      return MULS;
    case 6:      
      return MULU;
    case 7:
      return MULUS;
    case 8:
      return MULS_16;
    case 9:
      return MULU_16;
    case 10:
      return MULUS_16;
    case 11:
      return OR;
    case 12:
      return SEXT;
    case 13:
      if (mode == 32) {
	if (opcode[13] == '1') {
          return SLL;
        } else {
	  return SLLI;
        }
      } else {
        if (opcode[12] == '1') {
          return SLL;
        } else {
	  return SLLI;
        }
      }
    case 14:
      if (mode == 32) {
	if (opcode[13] == '1') {
          return SRA;
        } else {
	  return SRAI;
	}
      } else {
        if (opcode[12] == '1') {
          return SRA;
        } else {
	  return SRAI;
	}
      }
    case 15:
      if (mode == 32) {
	if (opcode[13] == '1') {
          return SRL;
        } else {
	  return SRLI;
	}
      } else {
        if (opcode[12] == '1') {
          return SRL;
        } else {
	  return SRLI;
	}
      }
    case 16:
      return SUB;
    case 17:
      return SUBU;
    case 18:
      return XOR;
    case 19:
      return MOV;
    case 20:
      return NOT;
    case 21:
      return DI;
    case 22:
      return EI;
    case 23:
      return RETI;
    case 24:
      return TRAP;
    case 25:
      return CMP;
    case 26:
      return EXBF;
    case 27:
      return JMPR;
    case 28:
      return SCON;
    case 29:
      return MULHI;
    case 30:
      return RCON;
    case 31:
      return RETU;
    case 32:
      return BC;
    case 33:
      return BEGT;
    case 34:
      return BELT;
    case 35:
      return BEQ;
    case 36:
      return BGT;
    case 37:
      return BLT;
    case 38:
      return BNE;
    case 39:
      return BNC;
    case 40:
      return ADDIU;
    case 41:
      return ANDI;
    case 42:
      return ORI;
    case 43:
      return SEXTI;
    case 44:
      return MOVFC;
    case 45:
      return ADDI;
    case 46:
      return MULI;
    case 47:
      return SWM;
    case 48:
      return EXB;
    case 49:
      return EXH;
    case 50:
      return LD;
    case 51:
      return CHRS;
    case 52:
      return ST;
    case 53:
      return JALR;
    case 54:
      return MOVTC;
    case 55:
      return CMPI;
    case 56:
      return JMP;
    case 57:
      return JAL;
    case 58:
      return NOP;
    case 59:
      return SCALL;
    case 60:
      return COP;
    case 61:
      return EXBFI;
    case 62:
      return LLI;
    case 63:
      return LUI;
    default:
      return NOP;
  }
}



//function which executes one instruction
bool coffee_model::execute_inst(INSTRUCTIONS inst, string bitinst,memory_models & mem) {
  switch (inst) {
    case ADD:

      return execute_add(bitinst);

    case ADDI:

      return execute_addi(bitinst);

    case ADDIU:

      return execute_addiu(bitinst);

    case ADDU :

      return execute_addu(bitinst);

    case AND :

      return execute_and(bitinst);

    case ANDI :

     return execute_andi(bitinst);

    case BC :

     return execute_bc(bitinst);
 
    case BEGT :

     return execute_begt(bitinst);

    case BELT :

     return execute_belt(bitinst);

    case BEQ :

     return execute_beq(bitinst);

    case BGT:

     return execute_bgt(bitinst);

    case BLT :

      return execute_blt(bitinst);

    case BNE :

      return execute_bne(bitinst);

    case BNC :

      return execute_bne(bitinst);

    case CHRS :

      return execute_chrs(bitinst);

    case CMP :

      return execute_cmp(bitinst);

    case CMPI :

      return execute_cmpi(bitinst);

    case CONB :

      return execute_conb(bitinst);

    case CONH :

      return execute_conh(bitinst);

    case COP :

      return execute_cop(bitinst);

    case DI :

      return execute_di();

    case EI :

      return execute_ei();

    case EXB :

      return execute_exb(bitinst);

    case EXBF:

      return execute_exbf(bitinst);

    case EXBFI :

      return execute_exbfi(bitinst);

    case EXH :

      return execute_exh(bitinst);

    case JAL:

      return execute_jal(bitinst);

    case JALR:

      return execute_jalr(bitinst);

    case JMP :

      return execute_jmp(bitinst);

    case JMPR :

      return execute_jmpr(bitinst);

    case LD :

      return execute_ld(bitinst,mem);

    case LLI :

      return execute_lli(bitinst);

    case LUI :

      return execute_lui(bitinst);

    case MOV :

     return execute_mov(bitinst);

    case MOVFC :

      return execute_movcf(bitinst);

    case MOVTC :

      return execute_movtc(bitinst);

    case MULHI :

      return execute_mulhi(bitinst);

    case MULI :

      return execute_muli(bitinst);

    case MULS :

      return execute_muls(bitinst);

    case MULS_16 :

      return execute_muls_16(bitinst);

    case MULU :

      return execute_mulu(bitinst);

    case MULU_16 :

      return execute_mulu_16(bitinst);

    case MULUS :

      return execute_mulus(bitinst);

    case MULUS_16 :

      return execute_mulus_16(bitinst);

    case NOP :

      return true;

    case NOT :

      return execute_not(bitinst);

    case OR :

      return execute_or(bitinst);

    case ORI :

      return execute_ori(bitinst);

    case RCON :

      return execute_rcon(bitinst);

    case RETI :

      return execute_reti();

    case RETU :

      return execute_retu();

    case SCALL:

      return execute_scall();

    case SCON :

      return execute_scon(bitinst);

    case SEXT :

      return execute_sext(bitinst);

    case SEXTI :

      return execute_sexti(bitinst);

    case SLL :

      return execute_sll(bitinst);

    case SLLI :

       return execute_slli(bitinst);

    case SRA :

      return execute_sra(bitinst);

    case SRAI :

      return execute_srai(bitinst);

    case SRL :

       return execute_srl(bitinst);

   case SRLI :

      return execute_srli(bitinst);

    case ST :

      return execute_st(bitinst,mem);

    case SUB :

      return execute_sub(bitinst);

    case SUBU :

      return execute_subu(bitinst);

    case SWM :

      return execute_swm(bitinst);

    case TRAP :
      return execute_trap(bitinst);

    case XOR :

      return execute_xor(bitinst);

  }
  return false;
}


// updating PC_value
void coffee_model::update_PC() {
  // If jump or exception
  if(jump_occured[0] == true) {
    PC = new_PC;
    jump_occured[0] = false;
    jump_occured[1] = false;
  }
  else{
    // If not mode change, increment PC normally.
    if(mode_change_delay > 2){
      if(mode == 32){
	PC += 4;
	// make sure alignment is OK.
	// PC = PC & 0xfffffffc;
      }
      else{
	PC += 2;
	// make sure alignment is OK.
	// PC = PC & 0xfffffffe;
      }
    }
    // If mode change is happening.
    else{
      // If mode is changed immediately.
      if(mode_change_delay == 2){
	// Switch to the new mode.
	mode = new_mode;
	// If new mode is 32-bit, align PC to 4 byte boundary.
	if(mode == 32){
	  // Increment PC.
	  PC += 2;
	  // If PC is not aligned properly after increment, increment
	  // it once more to make the alignment.
	  if((PC & 0x00000003) != 0){
	    PC += 2;
	  }
	  // PC = PC & 0xfffffffc; // Just to make sure.
	  // Update PSR
	  PSR = PSR | 0x00000008;
	  suser_reg[29] = suser_reg[29] | 0x00000008;
	}
	// If new mode is 16-bit, alignment goes right by default.
	else{
	  PC += 4;
	  // Update mode and PSR
	  PSR = PSR & 0xfffffff7;
	  suser_reg[29] = suser_reg[29] & 0xfffffff7;
	}
	// Increment mode_change_delay counter to turn mode change OFF.
	mode_change_delay += 2;
      }
      // If mode change is due to next cycles,
      // increment mode_change_delay counter and increment PC normally.
      else{
	if(mode == 32){
	  PC += 4;
	  // make sure alignment is OK.
	  // PC = PC & 0xfffffffc;
	}
	else{
	  PC += 2;
	  // make sure alignment is OK.
	  // PC = PC & 0xfffffffe;
	}	
	mode_change_delay += 1;
      }
    }
  }

  // Advance jump buffer (branch slot implementation)
  jump_occured[0] = jump_occured[1];
  jump_occured[1] = false;

}

 
//function that handles exceptions 
void coffee_model::handle_exception() {
    switch(ecause) {
      case I_ADDR_VIOL:
        PCB_REGS[0x15] = 0;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case UNK_OPC:
        PCB_REGS[0x15] = 1;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case ILLEG_OPCODE:
        PCB_REGS[0x15] = 2;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case MISS_ALIGN_JUMP:
        PCB_REGS[0x15] = 3;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case ETRAP:
        PCB_REGS[0x15] = trap_value;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case OFLOW:    
        PCB_REGS[0x15] = 6;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case D_ADDR_VIOL:
        PCB_REGS[0x15] = 7;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case J_ADDR_OFLOW:
        PCB_REGS[0x15] = 4;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
      case D_ADDR_OFLOW:
        PCB_REGS[0x15] = 8;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR; 
        break;
      case MISS_ALIGN_IADDR:
        PCB_REGS[0x15] = 5;
        PCB_REGS[0x16] = PC;
        PCB_REGS[0x17] = PSR;
        break;
    }
    // Set the following conditions:
    // Interrupts disabled, 32-bit mode, super user mode, and
    // super user register set for reading and writing.
    PSR = 0xe;
    update_PSR_variables(PSR);
    suser_reg[29] = PSR; // Make changes visible to the GUI.
    jump_occured[0] = true;
    new_PC = PCB_REGS[0x1e];
}


//initializing variables
coffee_model::coffee_model() {
  PC = 0;
  new_PC = 0;
  for(int i=0; i < 32 ; i++) {
    suser_reg[i] = 0;
    user_reg[i] = 0;
  }
  mulhi_result = 0;
  mode = 32;
  new_mode = 32;
  mode_change_delay = 5; // mode change OFF
  user_superuser = false;
  regtoread = 1;
  regtowrite = 1;
  interrupts_enabled = false;

  for ( int i = 0 ;i < 9; i++) {
    flag_reg[i] = 0;
    PC_STACK[i] = 0;
    PSR_STACK[i] = 0;
  }
  PSR = 14; // see specs
  suser_reg[29] = PSR;
  psr_update_delayed[2] = false;
  psr_update_delayed[1] = false;
  psr_update_delayed[0] = false;
  // write_to_register(29,PSR,1);
  write_to_register(30,9,1);
  PCB_REGS[0] =  0x10000;
  PCB_REGS[1] =  0x100ff;
  for ( int i= 2 ; i < 0xe ; i++ ) {
    PCB_REGS[i] = 1;
  }
  PCB_REGS[0x0e] = 0xfff;
  PCB_REGS[0x0f] = 0xfff;
  PCB_REGS[0x10] = 0;
  for ( int i= 0x11 ; i < 0x19 ; i++ ) {
    PCB_REGS[i] = 0;
  }
  PCB_REGS[0x19] = 0xffffffff;
  PCB_REGS[0x1a] = 0;
  PCB_REGS[0x1b] = 0xffffffff;
  PCB_REGS[0x1c] = 0x3;
  PCB_REGS[0x1d] = 1;
  PCB_REGS[0x1e] = 1;
  PCB_REGS[0x1f] = 0xfff;
  for ( int i= 0x20 ; i < 0x26 ; i++ ) {
    PCB_REGS[i] = 0;
  }
  PCB_REGS[0x26] = 0xffffffff;
  PCB_REGS[0x27] = 0xe;
  PCB_REGS[0x28] = 0;
  jump_occured[0] = false;
  jump_occured[1] = false;
  jmp_oflow = false;
  ecause = ETRAP;
  exception_occured = false;
  trap_value = 0;
  mode_change = 2;
  first_time = true;
  for(int i = 0 ; i<32; i++) {
      coprocessor0regs[i] = 0;
      coprocessor1regs[i] = 0;
      coprocessor2regs[i] = 0;
      coprocessor3regs[i] = 0;
  }
  timer0_value = 0; // Reset timer increments
  timer1_value = 0;
  timer0_stopped_to_max_count = false; // True if timer 0 has stopped to max count.
  timer1_stopped_to_max_count = false; // True if timer 1 has stopped to max count.
}

coffee_model::~coffee_model() {
}


//settting processor model to
//default values
void coffee_model::reset() {

  PC = 0;
  new_PC = 0;

  // Clear interrupt stacks.
  for(unsigned int j = 0; j < 12; j++){
    interrupt_pc_stack[j] = 0;
    interrupt_psr_stack[j] = 0;
    interrupt_cr0_stack[j] = 0;
    interrupt_number_stack[j] = 0;
  }
  // Execute the first instruction before updating the PC.
  skip_one_update = true;

  for(int i=0; i < 32 ; i++) {
    suser_reg[i] = 0;
    user_reg[i] = 0;
  }
  mulhi_result = 0;
  mode = 32;
  new_mode = 32;
  mode_change_delay = 5; // mode change OFF
  user_superuser = false;
  regtoread = 1;
  regtowrite = 1;
  interrupts_enabled = false;

  for ( int i = 0 ;i < 9; i++) {
    flag_reg[i] = 0;
    PC_STACK[i] = 0;
    PSR_STACK[i] = 0;
  }
  PSR = 14; // see specs
  suser_reg[29] = PSR;
  psr_update_delayed[2] = false;
  psr_update_delayed[1] = false;
  psr_update_delayed[0] = false;
  // write_to_register(29,PSR,1);
  write_to_register(30,9,1);
  PCB_REGS[0] =  0x10000;
  PCB_REGS[1] =  0x100ff;
  for ( int i= 2 ; i < 0xe ; i++ ) {
    PCB_REGS[i] = 1;
  }
  PCB_REGS[0x0e] = 0xfff;
  PCB_REGS[0x0f] = 0xfff;
  PCB_REGS[0x10] = 0;
  for ( int i= 0x11 ; i < 0x19 ; i++ ) {
    PCB_REGS[i] = 0;
  }
  PCB_REGS[0x19] = 0xffffffff;
  PCB_REGS[0x1a] = 0;
  PCB_REGS[0x1b] = 0xffffffff;
  PCB_REGS[0x1c] = 0x3;
  PCB_REGS[0x1d] = 1;
  PCB_REGS[0x1e] = 1;
  PCB_REGS[0x1f] = 0xfff;
  for ( int i= 0x20 ; i < 0x26 ; i++ ) {
    PCB_REGS[i] = 0;
  }
  PCB_REGS[0x26] = 0xffffffff;
  PCB_REGS[0x27] = 0xe;
  PCB_REGS[0x28] = 0;
  jump_occured[0] = false;
  jump_occured[1] = false;
  jmp_oflow = false;
  ecause = ETRAP;
  exception_occured = false;
  trap_value = 0;
  mode_change = 2;
  first_time = true;
  for(int i = 0 ; i<32; i++) {
      coprocessor0regs[i] = 0;
      coprocessor1regs[i] = 0;
      coprocessor2regs[i] = 0;
      coprocessor3regs[i] = 0;
  }
  timer0_value = 0; // Reset timer increments
  timer1_value = 0;
  timer0_stopped_to_max_count = false; // True if timer 0 has stopped to max count.
  timer1_stopped_to_max_count = false; // True if timer 1 has stopped to max count.
}


void coffee_model::update_psr(unsigned long psr) {
   update_PSR_variables(psr);

}


//function which steps processor one cycle forward
bool coffee_model::step(string& error_message, memory_models & memory){

  string temp = "";
  INSTRUCTIONS jep = NOP;
  bool except = false;

  // bool skip_one_update = false;

  unsigned long int_pend = PCB_REGS[0x12]; // Pending interrupts, active high
  unsigned long pending[12]; // pending interrupts in a table
  unsigned long int_mask = PCB_REGS[0x10]; // Interrupt mask, active low.
  unsigned long int_serv = PCB_REGS[0x11]; // Interrupts being served, active high.
  unsigned long int_pri[12];  // interrupt priorities
  unsigned int highest_pending_priority = 20; // highest pending priority
  unsigned int highest_pending_owner = 20; // The corresponding interrupt number
  unsigned int highest_served_priority = 20; // highest interrupt under service

  // Timers updated on each "cycle"
  update_timers();

  // Disable interrupts by skipping the interrupt test if the execution is already 
  // branching or jumping to somewhere.
  if(jump_occured[0] == false && jump_occured[1] == false){

    // Check if the execution sould go to an interrupt service routine.
    // If interrupts are enabled
    if((PSR & 16) > 0){

      // Apply the mask to the pending interrupts.
      int_pend = int_pend & int_mask;

      // If pending, unmasked interrupts are present.
      if(int_pend > 0){

	// Store the priority numbers of the interrupts into a table for easier access.
	int_pri[0] = (PCB_REGS[0x14] & 0x0000000F); // cop0 priority
	int_pri[1] = (PCB_REGS[0x14] & 0x000000F0); // cop1 priority
	int_pri[1] = int_pri[1] > 4;
	int_pri[2] = (PCB_REGS[0x14] & 0x00000F00); // cop2 priority
	int_pri[2] = int_pri[2] > 8;
	int_pri[3] = (PCB_REGS[0x14] & 0x0000F000); // cop3 priority
	int_pri[3] = int_pri[3] > 12;
	int_pri[4] = (PCB_REGS[0x13] & 0x0000000F); // int0 priority
	int_pri[5] = (PCB_REGS[0x13] & 0x000000F0); // int1 priority
	int_pri[5] = int_pri[5] > 4;
	int_pri[6] = (PCB_REGS[0x13] & 0x00000F00); // int2 priority
	int_pri[6] = int_pri[6] > 8;
	int_pri[7] = (PCB_REGS[0x13] & 0x0000F000); // int3 priority
	int_pri[7] = int_pri[7] > 12;
	int_pri[8] = (PCB_REGS[0x13] & 0x000F0000); // int4 priority
	int_pri[8] = int_pri[8] > 16;
	int_pri[9] = (PCB_REGS[0x13] & 0x00F00000); // int5 priority
	int_pri[9] = int_pri[9] > 20;
	int_pri[10] = (PCB_REGS[0x13] & 0x0F000000); // int6 priority
	int_pri[10] = int_pri[10] > 24;
	int_pri[11] = (PCB_REGS[0x13] & 0xF0000000); // int7 priority
	int_pri[11] = int_pri[11] > 28;

	// Mask to pick the right bit from the integer containing the 
	// indication bits of pending interrupts.
	unsigned int mask_number = 1;

	// Find the interrupt having the highest priority among the pending ones.
	for(unsigned int i = 0; i < 12; i++){
	  // If an interrupt is unmasked and pending
	  if( (int_pend & mask_number) > 0 ){
	    // If the interrupt contains the lowest priority number so far.
	    if( int_pri[i] < highest_pending_priority ){
	      highest_pending_priority = int_pri[i]; 
	      highest_pending_owner = i;
	    }
	  }
	  // Shift the mask bit left by one to get the next interrupt.
	  mask_number = mask_number << 1;
	}

	mask_number = 1;
	
	// Apply the mask also to the served interrupts.
	int_serv = int_serv & int_mask;      

	// Find the interrupt having the highest priority among the served ones
	for(unsigned int i = 0; i < 12; i++){
	  // If an interrupt is being served
	  if( (int_serv & mask_number) > 0 ){

	    // If the interrupt contains the lowest priority number so far.
	    if( int_pri[i] < highest_pending_priority ){
	      highest_served_priority = int_pri[i]; 
	    }
	  }
	  // Shift the mask bit left by one to get the next interrupt.
	  mask_number = mask_number << 1;
	}

	// If a pending interrupt has a higher priority than the one being served
	// switch to the correct interrupt service routine.
	if(highest_pending_priority < highest_served_priority){
	  switch_to_int_serv_routine(highest_pending_owner);
	  // Execute the instruction in the beginning of the interrupt service
	  // routine before advancing to the next one.
	  skip_one_update = true;
	}
      }
    }
  }

  // Don't update PC if skip_one_update is true.
  if(!skip_one_update){
    update_PC();
  }
  else{
    skip_one_update = false;
  }

  // checking if address value is valid
  if(check_instruction_address()) {
    // getting new instruction from the memory
    temp = memory.read_instr_mem(PC);

    if(temp == "ERROR") {
      error_message = "NOT A VALID ADDRESS FOR THE INSTRUCTION MEMORY";
      return false;
    }

    jep = which_instruction(temp);

    //executing one instruction
    if(check_if_execute(jep,temp)) {
      except = execute_inst(jep, temp, memory);
      if (except==false && (jep == LD || jep ==ST)) {
	error_message = "data memory problem";
      }
    }
    if(exception_occured){
      handle_exception();
      exception_occured = false;
    }
    
  }else{
    handle_exception();
    exception_occured = false;
  }

  // Shift the psr_update_delayed buffer left by one.
  // Once again an ugly, quick, bug fix hack.
  psr_update_delayed[2] = psr_update_delayed[1];
  psr_update_delayed[1] = psr_update_delayed[0];

  // Update PSR if the update of it was delayed.
  if(psr_update_delayed[2] == true)
  {
    regtowrite = 1;
    suser_reg[29] = suser_reg[30];
    PSR = suser_reg[30];
    update_PSR_variables(suser_reg[30]);
    psr_update_delayed[2] = false;
    psr_update_delayed[1] = false;
    psr_update_delayed[0] = false;
  }

  return true;
     
}


//THE FOLLOWING FUNCTIONS ARE FOR THE USE OF THE GUI 
//THEY GIVE NEEDED INFORMATION ABOUT THE STATUS OF THE MODEL
//AND VALUES INSIDE THE REGISTERS. BY USING THE POINTERS THE STATUS
//OF THE PROCESSOR CAN BE CHANGED


//function which returns instruction address value 
unsigned long coffee_model::give_instr_addr() {
  return PC;
}

//function which gives pointer to user register table
unsigned long* coffee_model::user_register_set(){
  unsigned long* temp = &user_reg[0];
  return temp;
} 
//function which gives pointer to superuser register table
unsigned long* coffee_model::superuser_register_set(){
  unsigned long* temp = &suser_reg[0];
  return temp;
} 

//function which gives pointer to ccb table
unsigned long* coffee_model::ccb(){
  unsigned long* temp = &PCB_REGS[0];
  return temp;
} 

//function which gives pointer to condition_reg_table
int* coffee_model::cond_reg_set() {
  int* temp = &flag_reg[0];
  return temp;
} 
    
//function which gives pointer to coprocessor register table
unsigned long* coffee_model::coprocessor0(){
  unsigned long* temp = &coprocessor0regs[0];
  return temp;
} 

//function which gives pointer to coprocessor register table
unsigned long* coffee_model::coprocessor1() {
  unsigned long* temp = &coprocessor1regs[0];
  return temp;
} 

//function which gives pointer to coprocessor register table
unsigned long* coffee_model::coprocessor2(){
  unsigned long* temp = &coprocessor2regs[0];
  return temp;
} 

//function which gives pointer to coprocessor register table
unsigned long* coffee_model::coprocessor3(){
  unsigned long* temp = &coprocessor3regs[0];
  return temp;
} 


// Function which receives an interrupt request.
void coffee_model::interrupt(std::string int_name){

  if (int_name == "cop_int0"){
    PCB_REGS[18] = PCB_REGS[18] | 1;
  }
  if (int_name == "cop_int1"){
    PCB_REGS[18] = PCB_REGS[18] | 2;
  }
  if (int_name == "cop_int2"){
    PCB_REGS[18] = PCB_REGS[18] | 4;
  }
  if (int_name == "cop_int3"){
    PCB_REGS[18] = PCB_REGS[18] | 8;
  }
  if (int_name == "int0"){
    PCB_REGS[18] = PCB_REGS[18] | 16;
  }
  if (int_name == "int1"){
    PCB_REGS[18] = PCB_REGS[18] | 32;
  }
  if (int_name == "int2"){
    PCB_REGS[18] = PCB_REGS[18] | 64;
  }
  if (int_name == "int3"){
    PCB_REGS[18] = PCB_REGS[18] | 128;
  }
  if (int_name == "int4"){
    PCB_REGS[18] = PCB_REGS[18] | 256;
  }
  if (int_name == "int5"){
    PCB_REGS[18] = PCB_REGS[18] | 512;
  }
  if (int_name == "int6"){
    PCB_REGS[18] = PCB_REGS[18] | 1024;
  }
  if (int_name == "int7"){
    PCB_REGS[18] = PCB_REGS[18] | 2048;
  }


}


// Function makes the switch to a certain interrupt service routine.
void coffee_model::switch_to_int_serv_routine(unsigned long int_num){

  unsigned long int_mode_il; // interrupt instruction length.
  unsigned long int_mode_um; // interrupt user mode

  switch (int_num){
  case 0:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 0;

    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;

    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = PCB_REGS[0x11] | 1; 
    // Clear the int_pend bit.
    PCB_REGS[0x12] = PCB_REGS[0x12] & 0xFFFFFFFE;
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x00000001);
    if(int_mode_il > 0){
      PSR = PSR | 0x00000008;
    }
    else{
      PSR = PSR & 0xFFFFFFF7;
    }

    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x00000001);
    // If user mode
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001); // UM bit to user mode
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{ // Super user mode
      PSR = (PSR & 0xFFFFFFFE); // UM bit to superuser mode
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }

    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x2];
    break;

  case 1:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 1;

    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 2);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffffd);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x00000002);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x00000002);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x3];
    break;

  case 2:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 2;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 4);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffffb);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x00000004);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x00000004);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x4];
    break;
    
  case 3:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 3;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 8);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffff7);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x00000008);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x00000008);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x5];
    break;
    
  case 4:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 4;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 16);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xffffffef);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 16);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 16);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x6];
    break;
    
  case 5:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 5;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 32);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xffffffdf);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 32);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 32);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x7];
    break;
    
  case 6:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 6;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 64);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xffffffbf);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 64);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 64);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x8];
    break;
    
  case 7:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 7;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 128);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xffffff7f);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 128);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 128);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0x9];
    break;
    
  case 8:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 8;
    
    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];
    
    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 256);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffeff);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 256);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 256);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0xa];
    break;
    
  case 9:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 9;

    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];

    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 0x200);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffdff);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x200);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x200);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0xb];
    break;

  case 10:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 10;

    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;

    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;
    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];

    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 0x400);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffffbff);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x400);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x400);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0xc];
    break;

  case 11:

    // Save interrupt number into the interrupt_number_stack for RETI-instruction
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_number_stack[i] = interrupt_number_stack[i-1];
    }
    interrupt_number_stack[0] = 11;

    // Save return address into the interrupt_pc_stack. 
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_pc_stack[i] = interrupt_pc_stack[i-1];
    }
    interrupt_pc_stack[0] = PC;
    // Save PSR to interrupt_psr_stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_psr_stack[i] = interrupt_psr_stack[i-1];
    }
    interrupt_psr_stack[0] = PSR;

    // Save CR0 to interrupt_cr0_stack
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_cr0_stack[i] = interrupt_cr0_stack[i-1];
    }
    interrupt_cr0_stack[0] = flag_reg[0];

    // Set the correct int_serv bit.
    PCB_REGS[0x11] = (PCB_REGS[0x11] | 0x800);
    // Clear the int_pend bit.
    PCB_REGS[0x12] = (PCB_REGS[0x12] & 0xfffff7ff);
    // Set the instruction length bit to the PSR
    int_mode_il = PCB_REGS[0xe];
    int_mode_il = (int_mode_il & 0x800);
    if(int_mode_il > 0){
      PSR = (PSR | 0x00000008);
    }
    else{
      PSR = (PSR & 0xFFFFFFF7);
    }
    // Set the user mode bit to the PSR
    int_mode_um = PCB_REGS[0xf];
    int_mode_um = (int_mode_um & 0x800);
    if(int_mode_um > 0){
      PSR = (PSR | 0x00000001);
      PSR = (PSR & 0xfffffff9); // User register set read and write
    }
    else{
      PSR = (PSR & 0xFFFFFFFE);
      PSR = (PSR | 0x00000006); // Superuser register set read and write
    }
    // Set the interrupt service routine start address to PC
    PC = PCB_REGS[0xd];
    break;

  }

    // Push jump_occured to the corresponding interrupt stack.
    for(unsigned int i = 11; i > 0 ; i--){
      interrupt_jump_occurred_stack[i][0] = interrupt_jump_occurred_stack[i-1][0];
      interrupt_jump_occurred_stack[i][1] = interrupt_jump_occurred_stack[i-1][1];
    }
    interrupt_jump_occurred_stack[0][0] = jump_occured[0];
    interrupt_jump_occurred_stack[0][1] = jump_occured[1];    

    // Clear the jump_occured[] condition, so that the interrupt is taken and not the jump.
    jump_occured[0] = false;
    jump_occured[1] = false;


  // Disable interrupts
  PSR = PSR & 0xffffffef;

  // Make PSR visible to the user.
  suser_reg[29] = PSR;

  // Update internal variables.
  update_PSR_variables(PSR);

  // Place the return address into the correct CCB register.
  PCB_REGS[0x26] = interrupt_pc_stack[0];

  // Place the return PSR into the correct CCB register.
  PCB_REGS[0x27] = interrupt_psr_stack[0];

  // Place the return CR0 into the correct CCB register.
  PCB_REGS[0x28] = interrupt_cr0_stack[0];

}


void coffee_model::update_timers(){

  // Define help variables.
  bool enable0; // enable timer 1
  bool enable1; // enable timer 2
  bool continuous0; // continuous mode for timer 1, true == continuous mode
  bool continuous1; // continuous mode for timer 2, true == continuous mode
  bool generate_interrupt0; // Timer 1 generate interrupt
  bool generate_interrupt1; // Timer 2 generate interrupt
  bool wdog0; // Timer 1 enable watchdog
  bool wdog1; // Timer 2 enable watchdog
  unsigned long interrupt_number0; // Interrupt number generated by the timer 0
  unsigned long interrupt_number1; // Interrupt number generated by the timer 1  
  unsigned long divider0; // "Clock cycle" divider for timer 0
  unsigned long divider1; // "Clock cycle" divider for timer 1

  // Initialize variables.
  unsigned long timer0conf = (PCB_REGS[0x25] & 0x0000ffff);
  unsigned long timer1conf = (PCB_REGS[0x25] & 0xffff0000);
  timer1conf = timer1conf >> 16;

  // initialize enables
  if( (timer0conf & 0x00008000) > 0 ){
    enable0 = true;
  }
  else{
    enable0 = false;
    timer0_value = 0; // Zero timer divider
  }
  if( (timer1conf & 0x00008000) > 0 ){
    enable1 = true;
  }
  else{
    enable1 = false;
    timer1_value = 0; // Zero timer divider
  }

  // initialize continuous_mode.
  if( (timer0conf & 0x00004000) > 0 ){
    continuous0 = true;
  }
  else{
    continuous0 = false;
  }
  if( (timer1conf & 0x00004000) > 0 ){
    continuous1 = true;
  }
  else{
    continuous1 = false;
  }

  // Initialize interrupt generation.
  if( (timer0conf & 0x00002000) > 0 ){
    generate_interrupt0 = true;
  }
  else{
    generate_interrupt0 = false;
  }
  if( (timer1conf & 0x00002000) > 0 ){
    generate_interrupt1 = true;
  }
  else{
    generate_interrupt1 = false;
  }
  
  // Initialize watchdogs.
  if( (timer0conf & 0x00001000) > 0 ){
    wdog0 = true;
  }
  else{
    wdog0 = false;
  }
  if( (timer1conf & 0x00001000) > 0 ){
    wdog1 = true;
  }
  else{
    wdog1 = false;
  }
  
  // Initialize interrupt numbers
  interrupt_number0 = (timer0conf & 0x00000700);
  interrupt_number0 = (interrupt_number0 >> 8);
  interrupt_number1 = (timer1conf & 0x00000700);
  interrupt_number1 = (interrupt_number1 >> 8);

  // Initialize dividers
  divider0 = timer0conf & 0x000000ff;
  divider1 = timer1conf & 0x000000ff;

  // Perform timer 0 functionality only if it is enabled.
  if(enable0)
  {
    // If counter is at the maximum value
    if(PCB_REGS[0x21] == PCB_REGS[0x22])
    {
      // If watchdog function is enabled.
      if(wdog0)
      {
	reset();
      }
      // If watchdog was not enabled
      else
      {
	// If timer has not stopped to max count
        if(!timer0_stopped_to_max_count)
	{
	  // If interrupt is generated
	  if(generate_interrupt0)
	  {
	    // Set the correct bit in the INT_PEND register.
	    switch(interrupt_number0)
	    {
	    case 0: // interrupt 0
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000010);
	      break;
	    case 1:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000020);
	      break;
	    case 2:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000040);
	      break;
	    case 3:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000080);
	      break;
	    case 4:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000100);
	      break;
	    case 5:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000200);
	      break;
	    case 6:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000400);
	      break;
	    case 7:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000800);
	      break;
	    }
	  }
	  // If continuous mode is set, reset the timer.
	  if(continuous0)
	  {
	    timer0_value = 0; // Set "clock cycle" counter to zero.
	    PCB_REGS[0x21] = 0; // Set timer to zero
	  }
	  // Else set a notice that the timer has stopped to max count.
	  else
	  {
	    timer0_stopped_to_max_count = true;
	  }
        }
	// If timer has stopped to maximum value.
	else
	{
	  // If continuous mode is set after the timer has stopped to max count, reset the timer.
	  if(continuous0)
	  {
	    timer0_value = 0; // Set "clock cycle" counter to zero.
	    PCB_REGS[0x21] = 0; // Set timer to zero
	  }
	}
      }
    }
    // If counter is not at the maximum value, unset the stopped_to_max_count flag and
    // increment the timer if necessary.
    else
    {
      timer0_stopped_to_max_count = false;

      // If "clock cycle" counter has reached the devider, increment the timer.
      if(divider0 >= timer0_value)
      {
	timer0_value = 0; // reset the "clock cycle" counter
	PCB_REGS[0x21] += 1; // Increment the timer.
      }
      // Else increment the "clock cycle" counter
      else
      {
	timer0_value += 1;
      }

    }
  }
  // If timer 0 is not enabled, make cycle_counter zero.
  else
  {
    timer0_value = 0;
  }


  // Perform timer 1 functionality only if it is enabled.
  if(enable1)
  {
    // If counter is at the maximum value
    if(PCB_REGS[0x23] == PCB_REGS[0x24])
    {
      // If watchdog function is enabled.
      if(wdog1)
      {
	reset();
      }
      // If watchdog was not enabled
      else
      {
	// If timer has not stopped to max count
        if(!timer1_stopped_to_max_count)
	{
	  // If interrupt is generated
	  if(generate_interrupt1)
	  {
	    // Set the correct bit in the INT_PEND register.
	    switch(interrupt_number1)
	    {
	    case 0: // interrupt 0
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000010);
	      break;
	    case 1:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000020);
	      break;
	    case 2:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000040);
	      break;
	    case 3:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000080);
	      break;
	    case 4:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000100);
	      break;
	    case 5:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000200);
	      break;
	    case 6:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000400);
	      break;
	    case 7:
	      PCB_REGS[0x12] = (PCB_REGS[0x12] | 0x00000800);
	      break;
	    }
	  }
	  // If continuous mode is set, reset the timer.
	  if(continuous1)
	  {
	    timer1_value = 0; // Set "clock cycle" counter to zero.
	    PCB_REGS[0x23] = 0; // Set timer to zero
	  }
	  // Else set a notice that the timer has stopped to max count.
	  else
	  {
	    timer1_stopped_to_max_count = true;
	  }
        }
	// If timer has stopped to maximum value.
	else
	{
	  // If continuous mode is set after the timer has stopped to max count, reset the timer.
	  if(continuous1)
	  {
	    timer1_value = 0; // Set "clock cycle" counter to zero.
	    PCB_REGS[0x23] = 0; // Set timer to zero
	  }
	}
      }
    }
    // If counter is not at the maximum value, unset the stopped_to_max_count flag and
    // increment the timer if necessary.
    else
    {
      timer1_stopped_to_max_count = false;

      // If "clock cycle" counter has reached the devider, increment the timer.
      if(divider1 >= timer1_value)
      {
	timer1_value = 0; // reset the "clock cycle" counter
	PCB_REGS[0x23] += 1; // Increment the timer.
      }
      // Else increment the "clock cycle" counter
      else
      {
	timer1_value += 1;
      }

    }
  }
  // If timer 0 is not enabled, make cycle_counter zero.
  else
  {
    timer1_value = 0;
  }
}


unsigned long coffee_model::get_pc()
{
  return PC;
}



  /*

  // JUST CRAP BELOW THIS POINT ...

    // If timer has reached maximum value and continuous mode is OFF and
    // the stop has already been acknowledged, don't do anything.
    if((PCB_REGS[0x21] == PCB_REGS[0x22]) && continuous0 && timer0_stopped_to_max_count)
    {
    }    
    else
    {
      // If timer has reached maximum value
      if(PCB_REGS[0x21] == PCB_REGS[0x22])
      {
	// If interrupt is to be generated
	if(generate_interrupt0)
	{
	}
      }
    }
    // Increment timer 0 "clock cycle" counter.
    timer0_value += 1;
    // Increment the timer only if necessary amount of "clock cycles" has passed.
    if(timer0_value >= divider0){
      PCB_REGS[0x21] += 1; // increment TMR0_CNT
      timer0_value = 0;
    }
    // If TMR0_CNT == TMR0_MAX_CNT
    if(PCB_REGS[0x21] == PCB_REGS[0x22] ){
      PCB_REGS[0x21] = 0; // TMR0_CNT = 0
      // If watchdog functionality is enabled, reset the core.
      if(wdog0){
	reset();
      }
      // If an interrupt has to be generated
      if(generate_interrupt0){
	// Select the correct interrupt
      }
    }
  }  
  // Increment both timer counters
  timer1_value += 1;
  // Increment actual timers if necessary amount of "clock cycles" has passed.
  if(timer0_value >= divider0){
    PCB_REGS[0x21] += 1; // increment TMR0_CNT
    timer0_value = 0;
  }
  if(timer1_value >= divider1){
    PCB_REGS[0x23] += 1; // increment TMR1_CNT
    timer0_value = 0;
  }
  // If TMR0_CNT == TMR0_MAX_CNT
  if(PCB_REGS[0x21] == PCB_REGS[0x22] ){
    PCB_REGS[0x21] = 0; // TMR0_CNT = 0
    // If watchdog functionality is enabled, reset the core.
    if(wdog0){
      reset();
    }
    // If an interrupt has to be generated
    if(generate_interrupt0){
      // Select the correct interrupt
    }
  }
  */










