#ifndef COFFEE_MODEL_H
#define COFFEE_MODEL_H

//coffee model class 

#include <string>

#include "memory_models.h"

enum INSTRUCTIONS {ADD, ADDI, ADDIU, ADDU, AND, ANDI, BC, BEGT, BELT, BEQ, BGT, BLT, BNE,BNC,CHRS, CMP, CMPI, CONB,
		   CONH, COP, DI, EI, EXB, EXBF, EXBFI, EXH, JAL, JALR, JMP, JMPR, LD, LLI, LUI, MOV, MOVFC,
                   MOVTC, MULHI, MULI, MULS, MULS_16, MULU, MULU_16, MULUS, MULUS_16, NOP, NOT, OR, ORI, RCON, 
		   RETI, RETU, SCALL, SCON, SEXT, SEXTI, SLL, SLLI, SRA, SRAI, SRL, SRLI, ST, SUB, SUBU, SWM, 
		   TRAP, XOR}; 
enum EXCEPTION_CAUSE {I_ADDR_VIOL, UNK_OPC, ILLEG_OPCODE, MISS_ALIGN_JUMP, ETRAP, OFLOW, D_ADDR_VIOL, J_ADDR_OFLOW, D_ADDR_OFLOW, MISS_ALIGN_IADDR};

class coffee_model {
  public:
    coffee_model();
    ~coffee_model();
    //function to reset processor model to default values
    void reset();

    //function to update psr register inside the model
    void update_psr(unsigned long psr);

    //function to excecute one instruction
    bool step(std::string& error_message, memory_models& memory);
    
    //function which gives current address
    unsigned long give_instr_addr();

    //function which gives pointer to user register table
    unsigned long* user_register_set();

    //function which gives pointer to superuser register table
    unsigned long* superuser_register_set();

    //function which gives pointer to ccb table
    unsigned long* ccb();

    //function which gives pointer to condition_reg_table
    int* cond_reg_set();
    
    //function which gives pointer to coprocessor0 register table
    unsigned long* coprocessor0();

    //function which gives pointer to coprocessor1 register table
    unsigned long* coprocessor1();

    //function which gives pointer to coprocessor2 register table
    unsigned long* coprocessor2();

    //function which gives pointer to coprocessor3 register table
    unsigned long* coprocessor3();

    // Function which receives an interrupt request.
    // Simulation of signalling multiple interrupts simultaneously can be 
    // done by calling this function several times with a different 
    // interrupt number before calling the step again.
    void interrupt(std::string int_name);

    // Returns the value of the program counter
    unsigned long get_pc();

 private:
    unsigned long PC;
    unsigned long new_PC;
    unsigned long suser_reg[32];
    unsigned long user_reg[32];
    int regtowrite;
    int regtoread;
    unsigned long mulhi_result;
    int mode;
    bool user_superuser;
    bool interrupts_enabled;
    int flag_reg[8];
    unsigned long PC_STACK[8];
    unsigned long PSR_STACK[8];
    unsigned long PSR;
    bool psr_update_delayed[3]; // Delayed PSR update by RETU-instruction.
    unsigned long PCB_REGS[41];
    // stack for interrupt return addresses.
    unsigned long interrupt_pc_stack[12];
    // Stack for PSR for nested interrupts.
    unsigned long interrupt_psr_stack[12];
    // Stack for condition register 0 for interrupts.
    int interrupt_cr0_stack[12];
    // interrupt number stack for RETI-instruction
    unsigned int interrupt_number_stack[12];
    // interrupt stack for delay slots (jumps)
    bool interrupt_jump_occurred_stack[12][2];
    bool skip_one_update;
    bool jump_occured[2];
    bool jmp_oflow;
    EXCEPTION_CAUSE ecause;
    bool exception_occured; 
    unsigned int trap_value;
    int mode_change;
    // 1 to 2 clock cycles delay for changing between 16 and 32 bit modes
    int mode_change_delay;
    int new_mode; // destination mode for SWM instruction.
    bool first_time;
    unsigned long coprocessor0regs[32];
    unsigned long coprocessor1regs[32];
    unsigned long coprocessor2regs[32];
    unsigned long coprocessor3regs[32];
    unsigned long timer0_value; // The amount of increments of timer 0
    unsigned long timer1_value; // The amount of increments of timer 1
    bool timer0_stopped_to_max_count; // True if timer 0 has stopped to max count.
    bool timer1_stopped_to_max_count; // True if timer 1 has stopped to max count.

    //function to convert string to decimal
    unsigned long convert_32bit_string_to_decimal(std::string value);

   //function to convert decima| to string
   std::string convert_32bit_decimal_to_string(unsigned long value); 
  
   //function to update conditional flags
   void update_flags(int C ,int Z ,int N, int reg_number);

   //function to update PSR variables
   void update_PSR_variables( unsigned long new_PSR); 
  
   //function which writes to register(tables suser_reg and user_reg)
   bool write_to_register(int number, unsigned long value, int reg_type);
  
   //function which reads from register(tables suser_reg and user_reg)
   unsigned long read_from_register(int number, int reg_type);

   //function to check the correctness of i-address
   bool check_instruction_address(); 

   //function to convert register number from string to integer format 
   int reg_number(std::string to_convert);

   // function that writes data from memory
   bool write_to_memory(unsigned long address, unsigned long writedata, memory_models & memmi);

   // function that reads data from memory
   bool read_from_memory(int memory_read_reg, unsigned long address, memory_models & memmi);

   //function to convert immediate from string to long format 
   long convert_signed_immediate(std::string to_convert);

   //function to convert immediate from string to unsigned long format 
   unsigned long convert_unsigned_immediate(std::string to_convert);

   // Checking if conditional execution is done
   bool conditional_execution(int creg, int condition);

   //Checking if the instruction is going to be executed
   bool check_if_execute(INSTRUCTIONS inst, std::string instruction);

   bool execute_add(std::string inst);

   //function that executes addi instruction 
   bool execute_addi(std::string inst);

   //function that executes addiu instruction 
   bool execute_addiu(std::string inst);

   //function that executes addu instruction 
   bool execute_addu(std::string inst);

   //function that executes and instruction 
   bool execute_and(std::string inst);

   //function that executes andi instruction 
   bool execute_andi(std::string inst);

   //function that executes or instruction 
   bool execute_or(std::string inst);

   //function that executes ori instruction 
   bool execute_ori(std::string inst);

   //function that executes xor instruction 
   bool execute_xor(std::string inst);

   //function that executes bc instruction 
   bool execute_bc(std::string inst);

   //function that executes begt instruction 
   bool execute_begt(std::string inst);
 
   //function that executes belt instruction 
   bool execute_belt(std::string inst);

   //function that executes beq instruction 
   bool execute_beq(std::string inst);

   //function that executes bgt instruction 
   bool execute_bgt(std::string inst);

   //function that executes blt instruction 
   bool execute_blt(std::string inst);

   //function that executes bne instruction 
   bool execute_bne(std::string inst);

   //function that executes bne instruction 
   bool execute_bnc(std::string inst);

   //function that executes chrs instruction 
   bool execute_chrs(std::string inst);

   //function that executes cmp instruction 
   bool execute_cmp(std::string inst);

   //function that executes cmpi instruction 
   bool execute_cmpi(std::string inst);

   //function that executes conb instruction 
   bool execute_conb(std::string inst);

   //function that executes conh instruction 
   bool execute_conh(std::string inst);

   //function that executes conh instruction 
   bool execute_cop(std::string inst);

   //function that executes di instruction 
   bool execute_di();

   //function that executes ei instruction 
   bool execute_ei();

   //function that executes exb instruction 
   bool execute_exb(std::string inst);

   //function that executes exbf instruction 
   bool execute_exbf(std::string inst);

   //function that executes exbfi instruction 
   bool execute_exbfi(std::string inst);

   //function that executes exh instruction 
   bool execute_exh(std::string inst);

   //function that executes jal instruction 
   bool execute_jal(std::string inst);

   //function that executes jalr instruction 
   bool execute_jalr(std::string inst);

   //function that executes jmp instruction 
   bool execute_jmp(std::string inst);

   //function that executes jmpr instruction 
   bool execute_jmpr(std::string inst);

   //function that executes ld instruction 
   bool execute_ld(std::string inst, memory_models & memclass);

   //function that executes lli instruction 
   bool execute_lli(std::string inst);

   //function that executes lui instruction 
   bool execute_lui(std::string inst);

   //function that executes mov instruction 
   bool execute_mov(std::string inst);

   //function that executes movtc instruction 
   bool execute_movtc(std::string inst);

   //function that executes movcf instruction 
   bool execute_movcf(std::string inst);

   //function that executes mulhi instruction 
   bool execute_mulhi(std::string inst);

   //function that executes muli instruction 
   bool execute_muli(std::string inst);

   //function that executes muls instruction 
   bool execute_muls(std::string inst);

   //function that executes muls_16 instruction 
   bool execute_muls_16(std::string inst);

   //function that executes mulu instruction 
   bool execute_mulu(std::string inst);

   //function that executes mulu_16 instruction 
   bool execute_mulu_16(std::string inst);

   //function that executes mulus instruction 
   bool execute_mulus(std::string inst);

   //function that executes mulus_16 instruction 
   bool execute_mulus_16(std::string inst);

   //function that executes not instruction 
   bool execute_not(std::string inst);

   //function that executes rcon instruction 
   bool execute_rcon(std::string inst);

   //function that executes reti instruction 
   bool execute_reti();

   //function that executes retu instruction 
   bool execute_retu();

   //function that executes scall instruction 
   bool execute_scall();

   //function that executes scon instruction 
   bool execute_scon(std::string inst);

   //function that executes sext instruction 
   bool execute_sext(std::string inst);

   //function that executes sexti instruction 
   bool execute_sexti(std::string inst);

   //function that executes sll instruction 
   bool execute_sll(std::string inst);

   //function that executes slli instruction 
   bool execute_slli(std::string inst);

   //function that executes sra instruction 
   bool execute_sra(std::string inst);

   //function that executes srai instruction 
   bool execute_srai(std::string inst);

   //function that executes srl instruction 
   bool execute_srl(std::string inst);

   //function that executes srli instruction 
   bool execute_srli(std::string inst);

   //function that executes st instruction 
   bool execute_st(std::string inst, memory_models & memclass);

   //function that executes sub instruction 
   bool execute_sub(std::string inst);

   //function that executes subu instruction 
   bool execute_subu(std::string inst);

   //function that executes swm instruction 
   bool execute_swm(std::string inst);

   //function that executes trap instruction 
   bool execute_trap(std::string inst);
 
   //function to find instruction from the binary std::string
   INSTRUCTIONS which_instruction(std::string opcode);

   //function which executes one instruction
   bool execute_inst(INSTRUCTIONS inst, std::string bitinst,memory_models & mem);

   //updating PC_value
   void update_PC();

   //function that handles exceptions 
   void handle_exception();

   // Function makes the switch to a certain interrupt service routine.
   void switch_to_int_serv_routine(unsigned long int_num);

   // Function to update timers.
   // Timers may tick forward or generate interrupts according to
   // TMR_CONF CCB register.
   void update_timers();


};



#endif
