// Memory Models of the ISS
// 2004  Tuukka Kasanko
// tuukka.kasanko@tut.fi
// Tampere University of Technology
// Institute of Digital and Computer Systems
// PO BOX 553
// FIN-33101 Tampere
// Finland
 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// -------------------------------------------------------------------------
// Revision history:
// 30.8.2004 Error checking added to function 
//           initialize_memories(std::string filename)
//           Does not check that there is enough memory when storing to
//           data or instruction memory map.
// --------------------------------------------------------------------------

#include <string>
#include <iostream>
#include <cstdlib>
#include <cmath>
#include <fstream>
#include <iomanip>
#include <algorithm>
#include "memory_models.h"


memory_models::memory_models(){
  i_mem_file_open = false;
  d_mem_file_open = false;

  data_memory.clear();
  instruction_memory.clear();

  last_updated_data_address = 0;
  last_updated_data = "";
}


memory_models::~memory_models(){
  if(is_instr_mem_file_open()){
    i_mem.close();
  }
  if(is_data_mem_file_open()){
    d_mem.close();
  }
}


std::string memory_models::read_instr_mem(unsigned long address){
  if(instruction_memory.count(address)){
    return instruction_memory[address];
  }
  else{
    return "ERROR";
  }
}


std::string memory_models::read_data_mem(unsigned long address){
  if(data_memory.count(address)){
    return data_memory[address];
  }
  else{
    return "ERROR";
  }
}


bool memory_models::write_data_mem(unsigned long address, std::string data){
  data_memory[address] = data;
  last_updated_data_address = address;
  last_updated_data = data;

   return true;
}


bool memory_models::is_instr_mem_file_open(){
  return i_mem_file_open;
}


bool memory_models::is_data_mem_file_open(){
  return d_mem_file_open;
}


// Initialize instruction memory
bool memory_models::initialize_inst_mem(std::string filename){
  std::ifstream infile(filename.c_str());
  if(infile){
    i_mem_file_open = true;
    std::string line = "";
    unsigned long addr = 0;
    while(getline(infile, line, '\n')){
      instruction_memory[addr] = line;
      addr += 4;
    }
    i_mem.close();
    i_mem_file_open = false;
    return true;
  }
  else{
    return false;
  }
}


// Initialize data memory
bool memory_models::initialize_data_mem(std::string filename){
  std::ifstream infile(filename.c_str());
  if(infile){
    d_mem_file_open = true;
    std::string line = "";
    unsigned long addr = 0;
    while(getline(infile, line, '\n')){
      data_memory[addr] = line;
      addr += 4;
    }
    d_mem.close();
    d_mem_file_open = false;
    return true;
  }
  else{
    return false;
  }
}


// Clear the instruction memory.
void memory_models::clear_instr_mem(){
  instruction_memory.clear();
}


// Clear the data memory.
void memory_models::clear_data_mem(){
  data_memory.clear();
}


// Returns a pointer to the data memory map
std::map<unsigned long,std::string>* memory_models::data_memory_map(){
  return &data_memory;
}


// Returns a pointer to the instruction memory map
std::map<unsigned long,std::string>* memory_models::instruction_memory_map(){
  return &instruction_memory;
}


// Returns last updated data address
unsigned long memory_models::gimme_last_updated_data_address(){
  return last_updated_data_address;
}


// Returns last updated data
std::string memory_models::gimme_last_updated_data(){
  return last_updated_data;
}


// Read in a COFF file and initialize memories from it.
bool memory_models::initialize_memories(std::string filename){

  std::ifstream infile( filename.c_str() );

  if(!infile){
    std::cerr << "Cannot open file " << filename  << std::endl;
    return false;
  }

  // variables corresponding to file header
  unsigned short f_magic_number;         // magic number             
  unsigned short f_num_of_sections;      // number of sections       
  unsigned long  f_time_and_date;        // time & date stamp       
  unsigned long  f_psymb_table;          // file pointer to symbol table
  unsigned long  f_num_of_entries;       // number of symbol table entries 
  unsigned short f_opt_header_size;      // sizeof(optional hdr)     
  unsigned short f_flags;                // flags                    

  // Character to read 8 bits from the file
  char ch;

  // Number of bytes read from the file
  unsigned long bytes_read = 0;

  // Reading file header
  // A really ugly way of doing it, but it works.

  // std::cout << "Reading the binary file." << std::endl;
  // std::cout << "bytes_read: " << bytes_read << std::endl;

  // getting the magic number
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the magic number." << std::endl;
    return false;
  }
  f_magic_number = (0x00ff & ch);
  f_magic_number = (f_magic_number << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the magic number." << std::endl;
    return false;
  }
  f_magic_number += (ch & 0x00ff);

  // std::cout << "f_magic_number: " << f_magic_number << std::endl;

  // Checking the magic number.
  if(f_magic_number != 0xC0FF){
    std::cerr << "This is not a valid COFF file." << std::endl;
    return false;
  }

  // getting number of sections
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of sections." << std::endl;
    return false;
  }
  f_num_of_sections = (0x00ff & ch);
  f_num_of_sections = (f_num_of_sections << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of sections." << std::endl;
    return false;
  }
  f_num_of_sections += (ch & 0x00ff);

  // std::cout << "f_num_of_sections: " << f_num_of_sections << std::endl;

  // getting time and date stamp
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the time stamp" << std::endl;
    return false;
  }
  f_time_and_date = (0x00ff & ch);
  f_time_and_date = (f_time_and_date << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the time stamp" << std::endl;
    return false;
  }
  f_time_and_date += (ch & 0x00ff);
  f_time_and_date = (f_time_and_date << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the time stamp" << std::endl;
    return false;
  }
  f_time_and_date += (ch & 0x00ff);
  f_time_and_date = (f_time_and_date << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the time stamp." << std::endl;
    return false;
  }
  f_time_and_date += (ch & 0x00ff);

  // std::cout << "f_time_and_date: " << f_time_and_date << std::endl;


  // getting symbol table pointer
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the symbol table pointer." << std::endl;
    return false;
  }
  f_psymb_table = (0x00ff & ch);
  f_psymb_table = (f_psymb_table << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the symbol table pointer." << std::endl;
    return false;
  }
  f_psymb_table += (ch & 0x00ff);
  f_psymb_table = (f_psymb_table << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the symbol table pointer." << std::endl;
    return false;
  }
  f_psymb_table += (ch & 0x00ff);
  f_psymb_table = (f_psymb_table << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the symbol table pointer." << std::endl;
    return false;
  }
  f_psymb_table += (ch & 0x00ff);

  // std::cout << "f_psymb_table: " << f_psymb_table << std::endl;


  // getting number of entries
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of entries." << std::endl;
    return false;
  }
  f_num_of_entries = (0x00ff & ch);
  f_num_of_entries = (f_num_of_entries << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of entries." << std::endl;
    return false;
  }
  f_num_of_entries += (ch & 0x00ff);
  f_num_of_entries = (f_num_of_entries << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of entries." << std::endl;
    return false;
  }
  f_num_of_entries += (ch & 0x00ff);
  f_num_of_entries = (f_num_of_entries << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the number of entries." << std::endl;
    return false;
  }
  f_num_of_entries += (ch & 0x00ff);

  // std::cout << "f_num_of_entries: " << f_num_of_entries << std::endl;

  // getting the size of the optional header
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the size of the optional header." << std::endl;
    return false;
  }
  f_opt_header_size = (0x00ff & ch);
  f_opt_header_size = (f_opt_header_size << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the size of the optional header." << std::endl;
    return false;
  }
  f_opt_header_size += (ch & 0x00ff);

  // std::cout << "f_opt_header_size: " << f_opt_header_size << std::endl;

  // getting flags
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the flags." << std::endl;
    return false;
  }
  f_flags = (0x00ff & ch);
  f_flags = (f_flags << 8);
  if(infile.good()){
    infile.get(ch);
  }
  else{
    std::cerr << "Unable to read the flags." << std::endl;
    return false;
  }
  f_flags += (ch & 0x00ff);  

  // std::cout << "f_flags: " << f_flags << std::endl;

  // Header contains 20 bytes
  bytes_read += 20;

  // Variables for a section header.

  // Section header = 40 bytes

  char           s_name[8];  // section name                     
  unsigned long  s_paddr;    // physical address
  unsigned long  s_vaddr;    // virtual address                  
  unsigned long  s_size;     // section size                     
  unsigned long  s_scnptr;   // file ptr to the raw data in the section 
  unsigned long  s_relptr;   // file ptr to relocation           
  unsigned long  s_lnnoptr;  // file ptr to line numbers         
  unsigned short s_nreloc;   // number of relocation entries     
  unsigned short s_nlnno;    // number of line number entries    
  unsigned long  s_flags;    // flags                            

  /*
  std::cout << "Sizes of variables, C++ is portable..." << std::endl;
  std::cout << "Size of unsigned int: " << sizeof(unsigned int) << std::endl;  
  std::cout << "Size of unsigned long: " << sizeof(unsigned long) << std::endl;  
  std::cout << "Size of unsigned short: " << sizeof(unsigned short) << std::endl;  
  */


  // Allocate memory for the headers of the sections
  // starting address of each section
  unsigned long* p_section_address = new unsigned long[f_num_of_sections];
  // sizes of each section
  unsigned long* p_section_size = new unsigned long[f_num_of_sections];
  // pointer to raw data for each section
  unsigned long* p_section_data = new unsigned long[f_num_of_sections];  
  // flags == type of each section
  unsigned long* p_section_flags = new unsigned long[f_num_of_sections];  
  // Section mode, main section, 16-bit subsection, 32-bit subsection
  unsigned long* p_section_mode = new unsigned long[f_num_of_sections];
  // Amount of subsections
  unsigned long* p_subsection_amount = new unsigned long[f_num_of_sections];
  // Contents of the section, RDATA, TEXT, DATA, BSS
  unsigned long* p_section_contents = new unsigned long[f_num_of_sections];


  if(p_section_address == 0 || p_section_size == 0 || p_section_data == 0 || p_section_flags == 0 ||
     p_section_mode == 0 || p_subsection_amount == 0 || p_section_contents == 0){
    std::cerr << "Unable to allocate memory for section headers." << std::endl;
    return false;    
  }

  // reading section headers if there are some
  for(unsigned short i = 0; i < f_num_of_sections; i++){

    // Getting the name of the section.
    for(unsigned short j = 0; j < 8; j++){
      if(infile.good()){
	infile.get(ch);
      }
      else{
	std::cerr << "Unable to read the section name." << std::endl;
	return false;
      }
      s_name[j] = ch;
    }

    // std::cout << "Reading a section header." << std::endl;
    
    // Getting the physical address of the section.
    infile.get(ch);
    s_paddr = (0x00ff & ch);
    s_paddr = (s_paddr << 8);
    infile.get(ch);
    s_paddr += (ch & 0x00ff);
    s_paddr = (s_paddr << 8);
    infile.get(ch);
    s_paddr += (ch & 0x00ff);
    s_paddr = (s_paddr << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the physical address of a section." << std::endl;
      return false;
    }
    s_paddr += (ch & 0x00ff);

    // std::cout << "s_paddr: " << s_paddr << std::endl;

    // Getting the virtual address of the section.
    infile.get(ch);
    s_vaddr = (0x00ff & ch);
    s_vaddr = (s_vaddr << 8);
    infile.get(ch);
    s_vaddr += (ch & 0x00ff);
    s_vaddr = (s_vaddr << 8);
    infile.get(ch);
    s_vaddr += (ch & 0x00ff);
    s_vaddr = (s_vaddr << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the virtual address of a section." << std::endl;
      return false;
    }
    s_vaddr += (ch & 0x00ff);

    // std::cout << "s_vaddr: " << s_vaddr << std::endl;

    // Getting the section size
    infile.get(ch);
    s_size = (0x00ff & ch);
    s_size = (s_size << 8);
    infile.get(ch);
    s_size += (ch & 0x00ff);
    s_size = (s_size << 8);
    infile.get(ch);
    s_size += (ch & 0x00ff);
    s_size = (s_size << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the size of a section." << std::endl;
      return false;
    }
    s_size += (ch & 0x00ff);

    // std::cout << "s_size: " << s_size << std::endl;

    // Getting the pointer to the data
    infile.get(ch);
    s_scnptr = (0x00ff & ch);
    s_scnptr = (s_scnptr << 8);
    infile.get(ch);
    s_scnptr += (ch & 0x00ff);
    s_scnptr = (s_scnptr << 8);
    infile.get(ch);
    s_scnptr += (ch & 0x00ff);
    s_scnptr = (s_scnptr << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the data pointer." << std::endl;
      return false;
    }
    s_scnptr += (ch & 0x00ff);

    // std::cout << "s_scnptr: " << s_scnptr << std::endl;

    // Getting the pointer to the relocation data
    infile.get(ch);
    s_relptr = (0x00ff & ch);
    s_relptr = (s_relptr << 8);
    infile.get(ch);
    s_relptr += (ch & 0x00ff);
    s_relptr = (s_relptr << 8);
    infile.get(ch);
    s_relptr += (ch & 0x00ff);
    s_relptr = (s_relptr << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the pointer to relocation data." << std::endl;
      return false;
    }
    s_relptr += (ch & 0x00ff);

    // std::cout << "s_relptr: " << s_relptr << std::endl;

    // Getting the pointer to line numbers
    infile.get(ch);
    s_lnnoptr = (0x00ff & ch);
    s_lnnoptr = (s_lnnoptr << 8);
    infile.get(ch);
    s_lnnoptr += (ch & 0x00ff);
    s_lnnoptr = (s_lnnoptr << 8);
    infile.get(ch);
    s_lnnoptr += (ch & 0x00ff);
    s_lnnoptr = (s_lnnoptr << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the pointer to line numbers." << std::endl;
      return false;
    }
    s_lnnoptr += (ch & 0x00ff);

    // std::cout << "s_lnnoptr: " << s_lnnoptr << std::endl;

    // Getting number of relocation entries
    infile.get(ch);
    s_nreloc = (0x00ff & ch);
    s_nreloc = (s_nreloc << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the number of relocation entries." << std::endl;
      return false;
    }
    s_nreloc += (ch & 0x00ff);

    // std::cout << "s_nreloc: " << s_nreloc << std::endl;

    // Getting the number of line numbers
    infile.get(ch);
    s_nlnno = (0x00ff & ch);
    s_nlnno = (s_nlnno << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the number of line numbers." << std::endl;
      return false;
    }
    s_nlnno += (ch & 0x00ff);

    // std::cout << "s_nlnno: " << s_nlnno << std::endl;

    // Getting the flags
    infile.get(ch);
    s_flags = (0x00ff & ch);
    s_flags = (s_flags << 8);
    infile.get(ch);
    s_flags += (ch & 0x00ff);
    s_flags = (s_flags << 8);
    infile.get(ch);
    s_flags += (ch & 0x00ff);
    s_flags = (s_flags << 8);
    if(infile.good()){
      infile.get(ch);
    }
    else{
      std::cerr << "Unable to read the flags." << std::endl;
      return false;
    }
    s_flags += (ch & 0x00ff);

    // std::cout << "s_flags: " << s_flags << std::endl;
    // std::cout << "Flags explained:" << std::endl;

    // Let's get the section mode.
    p_section_mode[i] = (s_flags & 0xFF000000);
    p_section_mode[i] = (p_section_mode[i] >> 24);

    /*
    if(p_section_mode[i] == 0){
	std::cout << "Main section, mode unknown." << std::endl;
    }
    if(p_section_mode[i] == 0x10){
	std::cout << "Subsection, 32-bit instruction encoding." << std::endl;
    }
    if(p_section_mode[i] == 0x01){
	std::cout << "Subsection, 16-bit instruction encoding." << std::endl;
    }
    */

    // Let's get the amount of subsections.
    p_subsection_amount[i] = s_flags;
    p_subsection_amount[i] = (p_subsection_amount[i] & 0x00FFFF00);
    p_subsection_amount[i] = (p_subsection_amount[i] >> 8);

    // std::cout << "Amount of subsections: " << p_subsection_amount[i] << std::endl;

    // Let's get the section contents description.
    p_section_contents[i] = (s_flags & 0x000000FF);

    /*
    if(p_section_contents[i] == 0x10){
	std::cout << "Read only data section." << std::endl;
    }
    if(p_section_contents[i] == 0x20){
	std::cout << "Text section." << std::endl;
    }
    if(p_section_contents[i] == 0x40){
	std::cout << "Initialized data section." << std::endl;
    }
    if(p_section_contents[i] == 0x80){
	std::cout << "Uninitialized data section." << std::endl;
    }
    */

    // Section header contains 40 bytes
    bytes_read += 40;

    p_section_address[i] = s_paddr;
    p_section_size[i] = s_size;
    p_section_data[i] = s_scnptr;
    p_section_flags[i] = s_flags;

  }

  // std::cout << "Bytes read so far: " << bytes_read << std::endl;

  // Now all the headers are read, and the next byte to read is the first
  // byte of "raw data" in the first section.

  // Go through all the sections, and skip the main sections.
  for(unsigned short i = 0; i < f_num_of_sections; i++){

    std::string one_word = "";
    unsigned long addr = 0;
    unsigned long raw_data_bytes_read = 0;

    // If the section is a main section and a code section, skip it, because
    // all the instructions are always stored into subsections in order to 
    // specify the instruction mode.
    if(p_section_mode[i] == 0 && p_section_contents[i] == 0x20){
      continue;
    }

    // std::cout << "p_section_mode[i]: " << p_section_mode[i] << std::endl;

    // 2 or 4 (bytes), depending on the mode.
    unsigned long word_length = 0;

    // if the section is a 16-bit section, read 2 bytes before
    // storing into the memory. Otherwise it is a 32-bit code section
    // or a data section and then 4 bytes can be read before storing 
    // the word into a memory.
    if(p_section_mode[i] == 0x01){
      word_length = 2;
    }
    else{
      word_length = 4;
    }

    // Read the whole data, section size was given in bytes.
    for(unsigned long j = 0; j < (p_section_size[i]/word_length); j++ ){

      // Read word_length amount of bytes.
      for(unsigned short k = 0; k < word_length; k++){

	if(infile.good()){
	  infile.get(ch);
	}
	else{
	  std::cerr << "Unable to read the contents of the section." << std::endl;
	  return false;
	}

	bytes_read += 1;
	raw_data_bytes_read += 1;

	// 8 bits in a byte
	for(unsigned short l = 0; l < 8; l++){
	  if((ch & 0x80) == 0){
	    one_word += "0";
	  }
	  else{
	    one_word += "1";
	  }
	  ch = ch << 1;
	}

      }

      if(word_length == 2){
	addr = p_section_address[i] + (raw_data_bytes_read - 2);
      }
      else{
	addr = p_section_address[i] + (raw_data_bytes_read - 4);
      }

      // Place the data to the correct memory.
      if(p_section_contents[i] == 0x20){
	instruction_memory[addr] = one_word;
      }
      else{
	data_memory[addr] = one_word;
      }

      // std::cout << one_word << std::endl;
      one_word = "";
    }
  }

  // Free dynamically allocated memory.
  delete[] p_section_address;
  delete[] p_section_size;
  delete[] p_section_data;
  delete[] p_section_flags;
  delete[] p_section_mode;
  delete[] p_subsection_amount;
  delete[] p_section_contents;

  infile.close();

  return true;

}

