// Menu Window of the ISS
// Copyright (C) 2004  Tuukka Kasanko
// tuukka.kasanko@tut.fi
// Tampere University of Technology
// Institute of Digital and Computer Systems
// PO BOX 553
// FIN-33101 Tampere
// Finland
 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// --------------------------------------------------------------------------
// Revision history:
// 3.9.2004   Continuous run added by introducing another thread.
//            Buttons disabled until a file is opened.
// 23.9.2004  Checking for a breakpoint added.
// 24.1.2005  Execution of the last instruction enabled.
//
// TODO:    - Quitting by clicking the quitting cross in the topwindow should
//            be disabled if the run-thread is active (running == true).
//            Could perhaps be done by overloading topwindow::hide?
//          - Currently the user cannot see that the run-thread has encountered
//            a breakpoint and stopped. Updating the windows in the run thread
//            is not possible, because the X does not support multiple threads.
//            It could perhaps be done by signalling stop.signal_clicked()
// --------------------------------------------------------------------------

#include "cond_registers.h"
#include "topwindow.h"
#include "inst_mem_window.h"
#include "copros_registers.h"
#include "ccb.h"
#include <gtkmm/stock.h>
#include <sigc++/class_slot.h>
#include <glibmm.h>
#include <string>
#include <iostream>


topwindow::topwindow():
  step("Step"), reset("Reset"), run("Run"), stop("Stop"), running(false), int0("int0"),
  int1("int1"), int2("int2"), int3("int3"), int4("int4"), int5("int5"), int6("int6"),
  int7("int7"), cop_int0("cop0int"), cop_int1("cop1int"), cop_int2("cop2int"), 
  cop_int3("cop3int")

{

  set_title("Menu Window");
  set_border_width(0);
  set_default_size(250, 50);

  // Add a vertical container to the window for menus and buttons.
  add(m_vbox);

  // File menu
  {
    Gtk::Menu::MenuList& menulist = File.items();
    menulist.push_back(Gtk::Menu_Helpers::MenuElem("_Open", Gtk::Menu::AccelKey("<control>o"),
						   SigC::slot(*this, &topwindow::on_open_file)));
    menulist.push_back(Gtk::Menu_Helpers::MenuElem("_Quit", Gtk::Menu::AccelKey("<control>q"),
						   SigC::slot(*this, &topwindow::on_quit)));
  }

  // Add File menu to the menubar
  m_MenuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("File", File));

  // Add menubar to the vertical container.
  m_vbox.pack_start(m_MenuBar, Gtk::PACK_SHRINK);

  // Add a horizontal container for the buttons to the vertical container.
  m_vbox.pack_start(m_hbox, Gtk::PACK_SHRINK);

  // Add the buttons
  step.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_button_clicked), "Step"));
  m_hbox.pack_start(step, Gtk::PACK_SHRINK);
  step.show();

  reset.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_button_clicked), "Reset"));
  m_hbox.pack_start(reset, Gtk::PACK_SHRINK);
  reset.show();

  run.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_button_clicked), "Run"));
  m_hbox.pack_start(run, Gtk::PACK_SHRINK);
  run.show();

  stop.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_button_clicked), "Stop"));
  m_hbox.pack_start(stop, Gtk::PACK_SHRINK);
  stop.show();

  // Add a horizontal container for the cop interrupt buttons to the vertical container.
  m_vbox.pack_start(m_hbox3, Gtk::PACK_SHRINK);

  // Add the coprocessor interrupt buttons
  cop_int0.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "cop_int0"));
  m_hbox3.pack_start(cop_int0, Gtk::PACK_SHRINK);
  cop_int0.show();

  // Add the coprocessor interrupt buttons
  cop_int1.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "cop_int1"));
  m_hbox3.pack_start(cop_int1, Gtk::PACK_SHRINK);
  cop_int1.show();

  // Add the coprocessor interrupt buttons
  cop_int2.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "cop_int2"));
  m_hbox3.pack_start(cop_int2, Gtk::PACK_SHRINK);
  cop_int2.show();

  // Add the coprocessor interrupt buttons
  cop_int3.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "cop_int3"));
  m_hbox3.pack_start(cop_int3, Gtk::PACK_SHRINK);
  cop_int3.show();

  // Add a horizontal container for the interrupt buttons to the vertical container.
  m_vbox.pack_start(m_hbox2, Gtk::PACK_SHRINK);

  // Add the interrupt buttons
  int0.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int0"));
  m_hbox2.pack_start(int0, Gtk::PACK_SHRINK);
  int0.show();

  // Add the interrupt buttons
  int1.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int1"));
  m_hbox2.pack_start(int1, Gtk::PACK_SHRINK);
  int1.show();

  // Add the interrupt buttons
  int2.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int2"));
  m_hbox2.pack_start(int2, Gtk::PACK_SHRINK);
  int2.show();

  // Add the interrupt buttons
  int3.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int3"));
  m_hbox2.pack_start(int3, Gtk::PACK_SHRINK);
  int3.show();

  // Add the interrupt buttons
  int4.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int4"));
  m_hbox2.pack_start(int4, Gtk::PACK_SHRINK);
  int4.show();

  // Add the interrupt buttons
  int5.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int5"));
  m_hbox2.pack_start(int5, Gtk::PACK_SHRINK);
  int5.show();

  // Add the interrupt buttons
  int6.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int6"));
  m_hbox2.pack_start(int6, Gtk::PACK_SHRINK);
  int6.show();

  // Add the interrupt buttons
  int7.signal_clicked().connect(SigC::bind<Glib::ustring>(SigC::slot(*this, &topwindow::on_int_button_clicked), "int7"));
  m_hbox2.pack_start(int7, Gtk::PACK_SHRINK);
  int7.show();

  step.set_sensitive(false);
  run.set_sensitive(false);
  reset.set_sensitive(false);
  stop.set_sensitive(false);

  int0.set_sensitive(false);
  int1.set_sensitive(false);
  int2.set_sensitive(false);
  int3.set_sensitive(false);
  int4.set_sensitive(false);
  int5.set_sensitive(false);
  int6.set_sensitive(false);
  int7.set_sensitive(false);
  cop_int0.set_sensitive(false);
  cop_int1.set_sensitive(false);
  cop_int2.set_sensitive(false);
  cop_int3.set_sensitive(false);

  m_hbox.show();

  user_register_window.show();
  super_user_register_window.show();
  cond_register_window.show();
  instruction_memory_window.show();
  copros_window.show();
  ccb_window.show();
  data_memory_window.show();

  show_all_children();

  previous_data_addr = 0;
  previous_data = "";

  boot_the_model();

}


topwindow::~topwindow()
{
}


void topwindow::on_button_clicked(Glib::ustring data)
{
  bool status = false;
  std::string error_message = "";
  unsigned long addr;
  std::string data_from_memory;

  if(data == "Step"){

    addr = core_model.give_instr_addr();    
    tmp_str = memories.read_instr_mem(addr);

    // If the address is not valid, prevent crashing by not updating the windows.
    if(tmp_str != "ERROR"){

      if(addr <= instruction_memory_window.addr_of_last_instr()){

	// If breakpoint present, do not step
	// if(instruction_memory_window.check_breakpoint(addr) == true){
	// return;
	// }

	status = core_model.step(error_message, memories);
	addr = core_model.give_instr_addr();

	// Prevent crashing by checking instruction address validity.
	tmp_str = memories.read_instr_mem(addr);
	if(tmp_str != "ERROR"){
	  if (status == true){
	    instruction_memory_window.select_instr(addr);
	    instruction_memory_window.make_visible();
	    update_register_windows();
	    addr = memories.gimme_last_updated_data_address();
	    data_from_memory = memories.gimme_last_updated_data();

	    // Update data memory window only if necessary
	    if(addr == previous_data_addr && 
	       data_from_memory == previous_data){
	    }
	    else{
	      // Update data memory window and set previous address and data
	      data_memory_window.store(addr, data_from_memory);
	      previous_data_addr = addr;	
	      previous_data = data_from_memory;
	    }
	  }
	  else{
	    std::cout << error_message << " at PC: " << addr << std::endl;
	  }
	}
	else{
	  std::cerr << "Instruction memory does not contain anything at address " << addr << std::endl;
	}
      }
    }
    // Inform the user from illegal instruction address.
    else{
      std::cerr << "Instruction memory does not contain anything at address " << addr << std::endl;
    }
  }

  if(data == "Reset"){
    boot_the_model();
    instruction_memory_window.select_instr(core_model.give_instr_addr());
    instruction_memory_window.make_visible();
  }

  if(data == "Stop"){
    running = false;

    // Update the windows so that the user can see the current state.
    addr = core_model.give_instr_addr();
    tmp_str = memories.read_instr_mem(addr);

    // If the PC points to an invalid instruction address, do not
    // try to highlight the instruction.
    if(tmp_str != "ERROR"){
      instruction_memory_window.select_instr(addr);
      instruction_memory_window.make_visible();
    }

    update_register_windows();
    // Clear the data memory window.
    data_memory_window.clear();
    // Put the contents of the data memory to the window.
    data_memory_window.initialize(memories.data_memory_map());

    // Make all buttons and menus functional.
    step.set_sensitive(true);
    run.set_sensitive(true);
    reset.set_sensitive(true);
    File.set_sensitive(true);
  }

  if(data == "Run"){
    running = true;

    // Set all buttons exept stop insensitive.
    step.set_sensitive(false);
    run.set_sensitive(false);
    reset.set_sensitive(false);

    // Set all menus insensitive
    File.set_sensitive(false);

    // Create a non-joinable thread which is deleted automatically on thread exit.
    Glib::Thread::create(SigC::slot_class(*this, &topwindow::continuous_run), false);

  }
}


// On interrupt button clicked.
void topwindow::on_int_button_clicked(Glib::ustring data){

  // Stop running before entering the interrupt service routine.
  // Running can be continued later if necessary, by setting running = true and
  // creating the running thread again.
  running = false;

  unsigned long addr;

  // Update the windows so that the user can see the current state.
  addr = core_model.give_instr_addr();
  instruction_memory_window.select_instr(addr);
  instruction_memory_window.make_visible();
  update_register_windows();

  // Clear the data memory window.
  data_memory_window.clear();
  // Put the contents of the data memory to the window.
  data_memory_window.initialize(memories.data_memory_map());

  // Make all buttons and menus functional.
  step.set_sensitive(true);
  run.set_sensitive(true);
  reset.set_sensitive(true);
  File.set_sensitive(true);

  core_model.interrupt(static_cast<std::string>(data));

  // Update all register windows. Extremely inefficient, as usual.
  update_register_windows();

}


void topwindow::boot_the_model()
{
  core_model.reset();
  super_user_register_window.reset();
  user_register_window.reset();
  update_register_windows();
}


void topwindow::on_open_file(){
  Gtk::FileSelection dialog("Open a COFF file");
  dialog.set_transient_for(*this);
  int result = dialog.run();

  // Handle the response:
  switch(result)
  {
  case(Gtk::RESPONSE_OK):
  {
    std::string filename = dialog.get_filename();

    // Clear instruction memory.
    memories.clear_instr_mem();
    // Clear instruction memory window
    instruction_memory_window.clear();
    // Clear data memory
    memories.clear_data_mem();
    // Clear data memory window
    data_memory_window.clear();

    if(memories.initialize_memories(filename)){
      // Get a pointer to the instruction memory map from the memory model
      // and pass it to the instruction memory window.
      instruction_memory_window.initialize(memories.instruction_memory_map());
      // Get a pointer to the data memory map from the memory model
      // and pass it to the data memory window.
      data_memory_window.initialize(memories.data_memory_map());

      // If everything went OK, enable all buttons.
      step.set_sensitive(true);
      run.set_sensitive(true);
      reset.set_sensitive(true);
      stop.set_sensitive(true);
      int0.set_sensitive(true);
      int1.set_sensitive(true);
      int2.set_sensitive(true);
      int3.set_sensitive(true);
      int4.set_sensitive(true);
      int5.set_sensitive(true);
      int6.set_sensitive(true);
      int7.set_sensitive(true);
      cop_int0.set_sensitive(true);
      cop_int1.set_sensitive(true);
      cop_int2.set_sensitive(true);
      cop_int3.set_sensitive(true);
      break;
    }
    else{
      std::cout << "The file " << filename  << " could not be opened for reading." << std::endl;
      break;
    }
    break;
  }
  case(Gtk::RESPONSE_CANCEL):
  {
    // std::cout << "Cancel clicked." << std::endl;
    break;
  }
  default:
  {
    std::cout << "Unexpected button clicked." << std::endl;
    break;
  }
  }
}


void topwindow::on_quit(){
  if(!running){
    hide();
  }
}


// Continuous run.
void topwindow::continuous_run(){
  bool status = false;
  std::string error_message = "";
  unsigned long addr;
  std::string data_from_memory;

  while(running){
    addr = core_model.give_instr_addr();    
    tmp_str = memories.read_instr_mem(addr);

    // If the address is not valid, prevent crashing by not trying to check the breakpoint.
    if(tmp_str != "ERROR"){
      if(addr < instruction_memory_window.addr_of_last_instr()){
	// If breakpoint present, stop running.
	if(instruction_memory_window.check_breakpoint(addr) == true){
	  running = false;
	}
	else{
	  status = core_model.step(error_message, memories);
	  if (status != true){
	    std::cout << error_message << " at PC: " << addr << std::endl;
	    running = false;
	  }
	}
      }
    }
    // If address was invalid, inform the user.
    else{
      running = false;
      std::cerr << "Instruction memory does not contain anything at address " << addr << std::endl;
    }
  }
}



void topwindow::update_register_windows()
{
  // Value of a register passed to a register window
  int value = 0;

  // value of a register passed to a window
  unsigned long reg_value;

  // pointer to the user register table
  unsigned long* p_user_table = core_model.user_register_set();

  // pointer to the superuser register table
  unsigned long* p_superuser_table = core_model.superuser_register_set();

  // pointer to condition register table
  int* p_cond_reg_table = core_model.cond_reg_set();

  // pointer to coprosessor0 register table
  unsigned long* p_copros0 = core_model.coprocessor0();

  // pointer to coprosessor1 register table
  unsigned long* p_copros1 = core_model.coprocessor1();

  // pointer to coprosessor2 register table
  unsigned long* p_copros2 = core_model.coprocessor2();

  // pointer to coprosessor3 register table
  unsigned long* p_copros3 = core_model.coprocessor3();

  // pointer to ccb
  unsigned long* p_ccb = core_model.ccb();

  // Updating user window
  for(unsigned int i = 0; i < 32; i++){
    value = static_cast<int>(*p_user_table);
    // value = *p_user_table;
    user_register_window.set_value(i, value);
    ++p_user_table;
  }

  // Updating superuser window
  for(unsigned int i = 0; i < 32; i++){
    value = static_cast<int>(*p_superuser_table);
    // value = *p_superuser_table;
    super_user_register_window.set_value(i, value);
    ++p_superuser_table;
  }

  // Updating condition register window
  for(unsigned int i = 0; i < 8; i++){
    value = *p_cond_reg_table;
    cond_register_window.set_value(i, value);
    ++p_cond_reg_table;
  }
  // Update PC to the condition register window.
  cond_register_window.set_value(8, core_model.get_pc());  

  // Updating coprosessor0 registers
  for(unsigned int i = 0; i < 32; i++){
    reg_value = *p_copros0;
    copros_window.set_value(0, i, reg_value);
    ++p_copros0;
  }

  // Updating coprosessor1 registers
  for(unsigned int i = 0; i < 32; i++){
    reg_value = *p_copros1;
    copros_window.set_value(1, i, reg_value);
    ++p_copros1;
  }

  // Updating coprosessor2 registers
  for(unsigned int i = 0; i < 32; i++){
    reg_value = *p_copros2;
    copros_window.set_value(2, i, reg_value);
    ++p_copros2;
  }

  // Updating coprosessor3 registers
  for(unsigned int i = 0; i < 32; i++){
    reg_value = *p_copros3;
    copros_window.set_value(3, i, reg_value);
    ++p_copros3;
  }

  // Update ccb window
  for(unsigned int i = 0; i < 41; i++){
    reg_value = *p_ccb;
    ccb_window.set_value(i, reg_value);
    ++p_ccb;
  }
}

