/*--------------------------------------------------------------------------*/
/*                    CUP-OS  scheduler routines                            */
/*              ---------------------------------------                     */
/*  Author            ---> Nuguru Susheel Raj                               */
/*  Modified by       ---> Nuguru Susheel Raj                               */
/*  Email-ID          ---> susheel.nuguru@tut.fi                            */
/*  Modified on       --->                                                  */
/*--------------------------------------------------------------------------*/

#include "errno.h"
#include "sched.h"

struct timer_list head_timer = INIT_TIMER;
extern struct task_struct *current;
extern struct task_struct idle_task;
int nr_running = 4;    //4 task initially in runqueue
unsigned int Global_Pid = 1;
int need_resched = 0; 
unsigned long volatile jiffies=0;
unsigned long int_mask_temp; // for enabling and disbling interrupts
                           // other than that used in interrupts and exceptions
//struct task_struct *task[NR_TASKS]={&init_task,};


struct task_struct init_task = INIT_TASK;


void check_priority(struct task_struct * p)
{
  if ( p->priority > current->priority )
    schedule();
}


void add_to_runqueue(struct task_struct * p)
{
#if 1	/* sanity tests */
	if (p->next_run || p->prev_run) {
//		printf("task already on run-queue\n");
		return;
	}
#endif
        nr_running++;
        p->timeout = WAIT_TIME;
        //p->counter = QUANTUM;
        (p->prev_run = init_task.prev_run)->next_run = p;
	p->next_run = &init_task;
	init_task.prev_run = p;
	check_priority(p);
	return;
}

inline void wake_up_process(struct task_struct * p)
{

  //asm("block_interrupts int_mask_temp\n\t");
      	p->state = TASK_RUNNING;
	if (!p->next_run)
	  {
                p->next_task = p->prev_task = NULL;
		add_to_runqueue(p);
	  }

  //asm("restore_interrupts int_mask_temp\n\t");
	return;
}


void process_waittimeout(unsigned long l_data)
{
	struct task_struct * p = (struct task_struct *) l_data;

	p->timeout = WAIT_TIME;
	wake_up_process(p); /*  defined in sched.h   */
	return;
}

struct signal_struct init_signals = INIT_SIGNALS;
struct task_struct *current=&init_task; /*points to currect task under execution */


extern inline void init_timer(struct timer_list * timer);



/* This verifies if the task is really in run queue so that it may be deleted.
   Note that the memory allocated to this is not removed, just its information
   regarding run queue is removed linking its neighbours in the queue together 
   also decrements the count of number of tasks in run queue  */

inline void del_from_runqueue(struct task_struct * p)
{
	struct task_struct *next = p->next_run;
	struct task_struct *prev = p->prev_run;

#if 1	/* sanity tests */
	if (!next || !prev) {
//		printf("task not on run-queue\n");
		return;
	}
#endif
	if (p == &init_task) 
          {
//	    printf("idle task may not sleep\n");
	    return;
	  }
	nr_running--;
        p->state = TASK_WAITING;
	next->prev_run = prev;
	prev->next_run = next;
	p->next_run = NULL;
	p->prev_run = NULL;
	if (p = current) 
	  {
	    need_resched = 1;
	    sys_getuid();
          }           // calling dummy system call which means calling the
	              // schedule() indirectly to make another high prioritized
	              // task = current
        
	return;
}

/* We dont actually need so many wait queues now but let me see how it ends up
   This is some what peculiar, we have only one queue when we wrote this 
   function.... versions of linux used some what complex waitqueues  */

void add_to_waitqueue(struct task_struct *to_wait)
{   struct task_struct *ptr = to_wait;

	    if (ptr->next_task != NULL || ptr->prev_task != NULL) 
             {
	       //	   printf("task already on wait-queue\n");
		   return;
	     }
       
        (ptr->prev_task = init_task.prev_task)->next_task = ptr;
	ptr->next_task = &init_task;
	init_task.prev_task = ptr;
	return;
    
} 


/* printf doesnt work unless it has been ported */

/* void Print_Runqueue( ) */
/* { */
/* 	struct task_struct* tmp_init = init_task.next_run; */


/* 	//Adding these printf bcos PrintTaskStruct doesnt print INIT */
/*        printf("PID:: 0,  PRIORITY:: 1 ----> INIT\n"); */

/*       while ( tmp_init != &init_task) */
/* 	{ */
/*               printf("PID:: %d,  PRIORITY:: %d\n",tmp_init->pid, tmp_init->priority); */
/* 	      tmp_init=tmp_init->next_run; */
/* 	}	;        */

/* } */
  

   
inline void move_last_runqueue(struct task_struct * p)
{
	struct task_struct *next = p->next_run;
	struct task_struct *prev = p->prev_run;

	/* remove from list */
	next->prev_run = prev;
	prev->next_run = next;
	/* add back to list */
	p->next_run = &init_task;
	prev = init_task.prev_run;
	init_task.prev_run = p;
	p->prev_run = prev;
	prev->next_run = p;
}

 inline int goodness(struct task_struct * p, struct task_struct * prev)
{
	int weight;

	/*
	 * Realtime process, select the first one on the
	 * runqueue (taking priorities within processes
	 * into account).
	 */
	if (p->policy != SCHED_OTHER)
		return 1000 + p->rt_priority;

	/*
	 * Give the process a first-approximation goodness value
	 * according to the number of clock-ticks it has left.
	 *
	 * Don't do any other calculations if the time slice is
	 * over..
	 */

	weight = p->counter;
	if (weight) {

		/* .. and a slight advantage to the current process */
		if (p == prev)
			weight += 1;
	}

	return weight;
}


#define TVN_BITS 6
#define TVR_BITS 8
#define TVN_SIZE (1 << TVN_BITS)
#define TVR_SIZE (1 << TVR_BITS)
#define TVN_MASK (TVN_SIZE - 1)
#define TVR_MASK (TVR_SIZE - 1)

#define SLOW_BUT_DEBUGGING_TIMERS 0

struct timer_vec {
        int index;
        struct timer_list *vec[TVN_SIZE];
};

struct timer_vec_root {
        int index;
        struct timer_list *vec[TVR_SIZE];
};

 struct timer_vec tv5 = { 0 };
 struct timer_vec tv4 = { 0 };
 struct timer_vec tv3 = { 0 };
 struct timer_vec tv2 = { 0 };
 struct timer_vec_root tv1 = { 0 };

 struct timer_vec * const tvecs[] = {
	(struct timer_vec *)&tv1, &tv2, &tv3, &tv4, &tv5
};

#define NOOF_TVECS (sizeof(tvecs) / sizeof(tvecs[0]))

 unsigned long timer_jiffies = 0;

 inline void insert_timer(struct timer_list *timer,
				struct timer_list **vec, int idx)
{
	if ((timer->next = vec[idx]))
		vec[idx]->prev = timer;
	vec[idx] = timer;
	timer->prev = (struct timer_list *)&vec[idx];
}

 inline void internal_add_timer(struct timer_list *timer)
{
	/*
	 * must be cli-ed when calling this
	 */
	unsigned long expires = timer->expires;
	unsigned long idx = expires - timer_jiffies;

	if (idx < TVR_SIZE) {
		int i = expires & TVR_MASK;
		insert_timer(timer, tv1.vec, i);
	} else if (idx < 1 << (TVR_BITS + TVN_BITS)) {
		int i = (expires >> TVR_BITS) & TVN_MASK;
		insert_timer(timer, tv2.vec, i);
	} else if (idx < 1 << (TVR_BITS + 2 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + TVN_BITS)) & TVN_MASK;
		insert_timer(timer, tv3.vec, i);
	} else if (idx < 1 << (TVR_BITS + 3 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + 2 * TVN_BITS)) & TVN_MASK;
		insert_timer(timer, tv4.vec, i);
	} else if (expires < timer_jiffies) {
		/* can happen if you add a timer with expires == jiffies,
		 * or you set a timer to go off in the past
		 */
		insert_timer(timer, tv1.vec, tv1.index);
	} else if (idx < 0xffffffffUL) {
		int i = (expires >> (TVR_BITS + 3 * TVN_BITS)) & TVN_MASK;
		insert_timer(timer, tv5.vec, i);
	} else {
		/* Can only get here on architectures with 64-bit jiffies */
		timer->next = timer->prev = timer;
	}
}

void add_timer(struct timer_list *timer)
{
	unsigned long flags;
	//save_flags(flags);
	//cli();
  //asm("block_interrupts int_mask_temp\n\t");
	internal_add_timer(timer);

	//restore_flags(flags);
  //asm("restore_interrupts int_mask_temp\n\t");
}

void add_to_timer_queue(struct timer_list *timer)
{

}

/* Very important change here ... greater the value of priority in the *
 * task_struct , heigher is its importance (Linus has opposite meaning)*
 * I dont wanna confuse myself */

void schedule(void)
{
	int c;
	struct task_struct * p;
	struct task_struct * prev, * next;
	unsigned long timeout = 0; //wait for so many ticks in timer wait queue

//	allow_interrupts();


	need_resched = 0;
	prev = current;
	//	cli();
  //asm("block_interrupts int_mask_temp\n\t");

	if (!prev->counter && prev->policy == SCHED_RR) {
		prev->counter = prev->priority;
		move_last_runqueue(prev);
	}
	switch (prev->state) {
		case TASK_INTERRUPTIBLE:
		  //signals and bottom halves not supported
		  //	if (prev->signal & ~prev->blocked)
		  //		goto makerunnable;
			timeout = prev->timeout;
			if (timeout && (timeout <= jiffies)) {
				prev->timeout = 0;
				timeout = 0;
		makerunnable:
				prev->state = TASK_RUNNING;
				break;
			}
		case TASK_RUNNING:
		                   break;
		default:
			del_from_runqueue(prev);
			prev = init_task.prev_run;
	}
	p = init_task.next_run;
	//sti();
  //asm("restore_interrupts int_mask_temp\n\t");
	
#define idle_task (&init_task)
	

/*
 * Note! there may appear new tasks on the run-queue during this, as
 * interrupts are enabled. However, they will be put on front of the
 * list, so our list starting at "p" is essentially fixed.
 */
/* this is the scheduler proper: */
	c = -1000;
	next = idle_task;
	while (p != idle_task) {
		int weight = goodness(p, prev);
		if (weight > c)
			c = weight, next = p;
		p = p->next_run;
	}

    /* if all runnable processes have "counter == 0", re-calculate counters */
	if (c == 0) {
		for_each_task(p)
			p->counter = (p->counter >> 1) +  p->priority;
	}

	if (prev != next) {
		struct timer_list *timer_pointer;
                timer_pointer = &prev->timer;
		if (timeout) {
			init_timer(timer_pointer);
			prev->timer.expires = timeout+jiffies;
			prev->timer.data = (unsigned long) prev;
			prev->timer.function = process_waittimeout;
			//add_timer(timer_pointer);
			//add_to_timer_queue(timer_pointer);
		} 

                current = next;

	}
	return;
}   


 inline int detach_timer(struct timer_list *timer)
{
	int ret = 0;
	struct timer_list *next, *prev;
	next = timer->next;
	prev = timer->prev;
	if (next) {
		next->prev = prev;
	}
	if (prev) {
		ret = 1;
		prev->next = next;
	}
	return ret;
}


extern int del_timer(struct timer_list * timer)
{
	int ret;
	unsigned long flags;
	//save_flags(flags);
	//cli();
  //asm("block_interrupts int_mask_temp\n\t");
	ret = detach_timer(timer);
	timer->next = timer->prev = 0;
	//restore_flags(flags);
  //asm("restore_interrupts int_mask_temp\n\t");
	return ret;
}

 inline void cascade_timers(struct timer_vec *tv)
{
        /* cascade all the timers from tv up one level */
        struct timer_list *timer;
        timer = tv->vec[tv->index];
        /*
         * We are removing _all_ timers from the list, so we don't  have to
         * detach them individually, just clear the list afterwards.
         */
        while (timer) {
                struct timer_list *tmp = timer;
                timer = timer->next;
                internal_add_timer(tmp);
        }
        tv->vec[tv->index] = NULL;
        tv->index = (tv->index + 1) & TVN_MASK;
}

 inline void run_timer_list(void)
{
  //cli();
  //asm("block_interrupts int_mask_temp\n\t");
	while ((long)(jiffies - timer_jiffies) >= 0) {
		struct timer_list *timer;
		if (!tv1.index) {
			int n = 1;
			do {
				cascade_timers(tvecs[n]);
			} while (tvecs[n]->index == 1 && ++n < NOOF_TVECS);
		}
		while ((timer = tv1.vec[tv1.index])) {
			void (*fn)(unsigned long) = timer->function;
			unsigned long data = timer->data;
			detach_timer(timer);
			timer->next = timer->prev = NULL;
			//sti();
  //asm("restore_interrupts int_mask_temp\n\t");
			fn(data);
			//cli();
  //asm("block_interrupts int_mask_temp\n\t");
		}
		++timer_jiffies; 
		tv1.index = (tv1.index + 1) & TVR_MASK;
	}
       //sti();
  //asm("restore_interrupts int_mask_temp\n\t");
}

unsigned long timer_active = 0;
struct timer_struct timer_table[32];

 inline void run_old_timers(void)
{
	struct timer_struct *tp;
	unsigned long mask;

	for (mask = 1, tp = timer_table+0 ; mask ; tp++,mask += mask) {
		if (mask > timer_active)
			break;
		if (!(mask & timer_active))
			continue;
		if (tp->expires > jiffies)
			continue;
		timer_active &= ~mask;
		tp->fn();
	}
}



/* This timer just removes the timer from the list of timers     */
/* Note if the timer is a newly created one then it is good practice to */
/* make its all pointers to NULL    */
  
inline void init_timer(struct timer_list * timer)
{
	timer->next = NULL;
	timer->prev = NULL;
}

/* update_process_times increments the number of ticks it had been since
   the system started and decrements the time quatum of current process,
   checks if current process is out of quantum, if yes then it enables
   rescheduling field in it, then it checks for any expired timers in the
   timer_list, if so put all the process whose timers expired in run_queue
   and then returns from the interrupt handler */
void update_process_times()
{   
      struct task_struct * p = current;   
      
        jiffies += 1;
	if(p->pid)
        {
            current->counter -= 1;
            if (current->counter <= 0)
            {
               current->counter = 0;
               need_resched = 1;
            }
            else
              need_resched = 0;
        }

	if(++current->syscall_called_timer == 200 && current->pid != 0)
	 {
	  current->state = TASK_STOPPED;
	
/*
 * someone told me that scheduler should be called only from system_call handler, keeping that in mind
 * scheduler is called indirectly here as dummy_idle_brain makes system_call which sets need_resched = 1
 * and thus calling the scheduler, If someone feels this is very bad IDEA (as i am feeling now) then just
 * replace dummy_idle_brain() with schedule(). In fact this replacement would save hundreds of instructions.
 * Ok leaving choice to the developers :-)
 */
	  dummy_idle_brain();
	 }
          
        get_expired_timer();

	if((current->pid == 0) && (nr_running != 0))
	  {
	    need_resched=1;
            dummy_idle_brain(); 
	  }
}

/*  This one searches the whole timer list for expired timers */
/*  and if expired put the task in the run queue              */

void get_expired_timer()
{
  extern struct timer_list head_timer;
  struct timer_list *p = head_timer.next;
       while(p != &head_timer)
      {
	if((p->expires - jiffies) <= 0)
	  { 
	    process_waittimeout(p->data);  /* declared in sched.h */
            del_timer(p);
          }
        p = p->next; 
      }
      return;
}



/*
 * For backwards compatibility?  This can be done in libc so Alpha
 * and all newer ports shouldn't need it.
 */
int sys_pause(void)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule();
	return -ERESTART;
}

int sys_getpid(void)
{
	return current->pid;
}

int sys_getppid(void)
{
  if(current->p_opptr->pid)
    return current->p_opptr->pid;
  else
    return -10;//current->p_opptr->pid;
}

int sys_getuid(void)
{
	return ENOSYS;
}

int sys_geteuid(void)
{
	return ENOSYS;
}

int sys_getgid(void)
{
	return ENOSYS;
}

int sys_getegid(void)
{
	return ENOSYS;
}

/*
 * This has been replaced by sys_setpriority.  Maybe it should be
 * moved into the arch dependent tree for those ports that require
 * it for backward compatibility?
 */
int sys_nice(int increment)
{
	unsigned long newprio;
	int increase = 0;

	newprio = increment;
	if (increment < 0) {
	  //	if (!suser())          everyone is super user right now
	      //	return -EPERM;
		newprio = -increment;
		increase = 1;
	}
	if (newprio > 40)
		newprio = 40;
	/*
	 * do a "normalization" of the priority (traditionally
   	 * unix nice values are -20..20, linux doesn't really
	 * use that kind of thing, but uses the length of the
	 * timeslice instead (default 150 msec). The rounding is
	 * why we want to avoid negative values.
	 */
	newprio = (newprio * DEF_PRIORITY + 10) / 20;
	increment = newprio;
	if (increase)
		increment = -increment;
	newprio = current->priority - increment;
	if ((signed) newprio < 1)
		newprio = 1;
	if (newprio > DEF_PRIORITY*2)
		newprio = DEF_PRIORITY*2;
	current->priority = newprio;
	return 0;
}

 struct task_struct *find_process_by_pid(pid_t pid) {
	struct task_struct *p, *q;

	if (pid == 0)
		p = current;
	else {
		p = 0;
		for_each_task(q) {
			if (q && q->pid == pid) {
				p = q;
				break;
			}
		}
	}
	return p;
}

/* Note here third argument changed a bit from Uclinux codes to suit here */
int setscheduler(pid_t pid, int policy, 
			int  sched_priority)
{
  //	int error;
  //	struct sched_param lp;
	struct task_struct *p;

	if (pid < 0)
		return -EINVAL;


	p = find_process_by_pid(pid);
	if (!p)
		return -ESRCH;
			
	if (policy < 0)
		policy = p->policy;

	else if (policy != SCHED_FIFO && policy != SCHED_RR &&
		 policy != SCHED_OTHER)
		return -EINVAL;
	
	/*
	 * Valid priorities for SCHED_FIFO and SCHED_RR are 1..99, valid
	 * priority for SCHED_OTHER is 0.
	 */

	if (sched_priority < 0 || sched_priority > 99)
		return -EINVAL;

	if ((policy == SCHED_OTHER) != (sched_priority == 0))
		return -EINVAL;


	p->policy = policy;
	p->rt_priority = sched_priority;
	//	cli();
  //asm("block_interrupts int_mask_temp\n\t");
	if (p->next_run)
		move_last_runqueue(p);
	//	sti();
  //asm("restore_interrupts int_mask_temp\n\t");
	need_resched = 1;
	return 0;
}

int sys_sched_setscheduler(pid_t pid, int policy, 
				      int sched_priority)
{
	return setscheduler(pid, policy, sched_priority);
}

int sys_sched_setparam(pid_t pid,int sched_priority)
{
	return setscheduler(pid, -1, sched_priority);
}


int sys_sched_getscheduler(pid_t pid)
{
	struct task_struct *p;

	if (pid < 0)
		return -EINVAL;

	p = find_process_by_pid(pid);
	if (!p)
		return -ESRCH;
			
	return p->policy;
}

int sys_sched_yield(void)
{
  //	cli();
  //asm("block_interrupts int_mask_temp\n\t");
	move_last_runqueue(current);
	current->counter = 0;
	need_resched = 1;
 //	sti();
  //asm("restore_interrupts int_mask_temp\n\t");
	return 1;
}

int sys_sched_get_priority_max(int policy)
{
	switch (policy) {
	      case SCHED_FIFO:
	      case SCHED_RR:
		return 99;
	      case SCHED_OTHER:
		return 0;
	}

	return -EINVAL;
}

int sys_sched_get_priority_min(int policy)
{
	switch (policy) {
	      case SCHED_FIFO:
	      case SCHED_RR:
		return 1;
	      case SCHED_OTHER:
		return 0;
	}

	return -EINVAL;
}

int sys_dummy_idle_brain()
{
  need_resched = 1;  /*this is called only when init_task is running or when the kernel decided to replace current task with other
                      * for the reason other than that the current has its time quantum expired and when returning from this systemcall
		      * need_resched will make scheduler check if there are any running tasks on the run queue */
  return 0;
}

// unimplemented syscalls when called the kernel executes the below function
int nosys()
{
  return ENOSYS;
}
